package ru.yandex.direct.core.entity.hypergeo.validation;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegmentDetails;
import ru.yandex.direct.core.entity.hypergeo.model.HyperPoint;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoSegmentRepository;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.hypergeo.model.GeoSegmentType.CONDITION;
import static ru.yandex.direct.core.entity.vcard.service.validation.PointOnMapValidator.LATITUDE_MAX;
import static ru.yandex.direct.core.entity.vcard.service.validation.PointOnMapValidator.LATITUDE_MIN;
import static ru.yandex.direct.core.entity.vcard.service.validation.PointOnMapValidator.LONGITUDE_MAX;
import static ru.yandex.direct.core.entity.vcard.service.validation.PointOnMapValidator.LONGITUDE_MIN;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
@ParametersAreNonnullByDefault
public class HyperGeoSegmentValidationService {

    private static final int MIN_POINTS_COUNT_IN_GEO_SEGMENT = 1;

    static final int MIN_GEO_SEGMENT_CIRCLE_RADIUS = 500;
    static final int MAX_GEO_SEGMENT_CIRCLE_RADIUS = 10000;

    public static final int MIN_GEO_SEGMENT_PERIOD_LENGTH = 1;
    public static final int MAX_GEO_SEGMENT_PERIOD_LENGTH = 90;

    public static final int MIN_GEO_SEGMENT_TIMES_QUANTITY = 1;
    public static final int MAX_GEO_SEGMENT_TIMES_QUANTITY = 90;

    private final HyperGeoSegmentRepository hyperGeoSegmentRepository;

    @Autowired
    public HyperGeoSegmentValidationService(HyperGeoSegmentRepository hyperGeoSegmentRepository) {
        this.hyperGeoSegmentRepository = hyperGeoSegmentRepository;
    }

    public ValidationResult<List<HyperGeoSegment>, Defect> validateHyperGeoSegments(List<HyperGeoSegment> hyperGeoSegments) {
        return ListValidationBuilder.of(hyperGeoSegments, Defect.class)
                .checkEachBy(this::validateHyperGeoSegment)
                .getResult();
    }

    private ValidationResult<HyperGeoSegment, Defect> validateHyperGeoSegment(HyperGeoSegment hyperGeoSegment) {
        ModelItemValidationBuilder<HyperGeoSegment> ivb = ModelItemValidationBuilder.of(hyperGeoSegment);

        ivb.item(HyperGeoSegment.SEGMENT_DETAILS)
                .check(notNull())
                .checkBy(this::validateHyperGeoSegmentDetails, When.isValid());

        return ivb.getResult();
    }

    private ValidationResult<HyperGeoSegmentDetails, Defect> validateHyperGeoSegmentDetails(HyperGeoSegmentDetails hyperGeoSegmentDetails) {
        ModelItemValidationBuilder<HyperGeoSegmentDetails> ivb =
                ModelItemValidationBuilder.of(hyperGeoSegmentDetails);
        boolean isConditionSegmentType = hyperGeoSegmentDetails.getGeoSegmentType() == CONDITION;

        ivb.item(HyperGeoSegmentDetails.RADIUS)
                .check(notNull())
                .check(notLessThan(MIN_GEO_SEGMENT_CIRCLE_RADIUS))
                .check(notGreaterThan(MAX_GEO_SEGMENT_CIRCLE_RADIUS));

        ivb.item(HyperGeoSegmentDetails.SEGMENT_NAME)
                .check(notBlank());

        ivb.list(HyperGeoSegmentDetails.POINTS)
                .check(notNull())
                .check(minListSize(MIN_POINTS_COUNT_IN_GEO_SEGMENT))
                .checkEachBy(this::validateHyperPoint);

        ivb.item(HyperGeoSegmentDetails.GEO_SEGMENT_TYPE)
                .check(notNull());

        ivb.item(HyperGeoSegmentDetails.PERIOD_LENGTH)
                .check(isNull(), When.isFalse(isConditionSegmentType))
                .check(notNull(), When.isTrue(isConditionSegmentType))
                .check(inRange(MIN_GEO_SEGMENT_PERIOD_LENGTH, MAX_GEO_SEGMENT_PERIOD_LENGTH), When.isValid());

        ivb.item(HyperGeoSegmentDetails.TIMES_QUANTITY)
                .check(isNull(), When.isFalse(isConditionSegmentType))
                .check(notNull(), When.isTrue(isConditionSegmentType))
                .check(inRange(MIN_GEO_SEGMENT_TIMES_QUANTITY, MAX_GEO_SEGMENT_TIMES_QUANTITY), When.isValid());

        return ivb.getResult();
    }

    private ValidationResult<HyperPoint, Defect> validateHyperPoint(HyperPoint hyperPoint) {
        ModelItemValidationBuilder<HyperPoint> ivb = ModelItemValidationBuilder.of(hyperPoint);

        ivb.item(HyperPoint.LATITUDE)
                .check(notNull())
                .check(notLessThan(LATITUDE_MIN.doubleValue()))
                .check(notGreaterThan(LATITUDE_MAX.doubleValue()));
        ivb.item(HyperPoint.LONGITUDE)
                .check(notNull())
                .check(notLessThan(LONGITUDE_MIN.doubleValue()))
                .check(notGreaterThan(LONGITUDE_MAX.doubleValue()));

        return ivb.getResult();
    }

    public ValidationResult<List<Long>, Defect> validateDelete(int shard, List<Long> hyperGeoSegmentIds) {
        var unusedHyperGeoSegmentIds = hyperGeoSegmentRepository.getUnusedHyperGeoSegmentIds(shard, hyperGeoSegmentIds);

        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(hyperGeoSegmentIds);
        lvb
                .check(notNull())
                .checkEach(notNull(), When.isValid())
                .checkEach(validId(), When.isValid())
                .checkEach(inSet(unusedHyperGeoSegmentIds), When.isValid());
        return lvb.getResult();
    }
}
