package ru.yandex.direct.core.entity.hypergeo.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSimple;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoSegmentRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class HyperGeoValidationService {

    private static final int MIN_GEO_SEGMENTS_COUNT_IN_HYPER_GEO = 1;

    private final HyperGeoSegmentRepository hyperGeoSegmentRepository;
    private final HyperGeoRepository hyperGeoRepository;

    @Autowired
    public HyperGeoValidationService(HyperGeoSegmentRepository hyperGeoSegmentRepository,
                                     HyperGeoRepository hyperGeoRepository) {
        this.hyperGeoSegmentRepository = hyperGeoSegmentRepository;
        this.hyperGeoRepository = hyperGeoRepository;
    }

    public ValidationResult<List<HyperGeoSimple>, Defect> preValidateHyperGeos(List<HyperGeoSimple> hyperGeos) {
        ListValidationBuilder<HyperGeoSimple, Defect> lvb = ListValidationBuilder.of(hyperGeos, Defect.class);

        lvb.checkEachBy(this::preValidateHyperGeo);

        return lvb.getResult();
    }

    private ValidationResult<HyperGeoSimple, Defect> preValidateHyperGeo(HyperGeoSimple hyperGeo) {
        ModelItemValidationBuilder<HyperGeoSimple> ivb = ModelItemValidationBuilder.of(hyperGeo);

        ivb.list(HyperGeoSimple.HYPER_GEO_SEGMENT_IDS)
                .check(minListSize(MIN_GEO_SEGMENTS_COUNT_IN_HYPER_GEO))
                .checkEach(validId());
        ivb.item(HyperGeoSimple.NAME)
                .check(notBlank());

        return ivb.getResult();
    }

    public ValidationResult<List<HyperGeoSimple>, Defect> validateHyperGeos(int shard, ClientId clientId,
                                                                            List<HyperGeoSimple> hyperGeos) {
        ListValidationBuilder<HyperGeoSimple, Defect> lvb = ListValidationBuilder.of(hyperGeos, Defect.class);

        Set<Long> hyperGeoSegmentIds = StreamEx.of(hyperGeos)
                .flatMap(hyperGeo -> StreamEx.of(hyperGeo.getHyperGeoSegmentIds()))
                .toSet();
        Map<Long, HyperGeoSegment> hyperGeoSegmentById =
                hyperGeoSegmentRepository.getHyperGeoSegmentById(shard, clientId, hyperGeoSegmentIds);

        lvb.checkEachBy(hyperGeo -> validateHyperGeo(hyperGeo, hyperGeoSegmentById.keySet()));

        return lvb.getResult();
    }

    private ValidationResult<HyperGeoSimple, Defect> validateHyperGeo(HyperGeoSimple hyperGeo,
                                                                      Set<Long> existingHyperGeoSegmentIds) {
        ModelItemValidationBuilder<HyperGeoSimple> ivb = ModelItemValidationBuilder.of(hyperGeo);

        ivb.list(HyperGeoSimple.HYPER_GEO_SEGMENT_IDS)
                .checkEach(inSet(existingHyperGeoSegmentIds), objectNotFound());

        return ivb.getResult();
    }

    public ValidationResult<List<Long>, Defect> validateDelete(DSLContext context, @Nullable ClientId clientId,
                                                               List<Long> hyperGeoIds) {
        var unusedHyperGeoIds = hyperGeoRepository.getUnusedHyperGeoIds(context, clientId, hyperGeoIds);

        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(hyperGeoIds);
        lvb
                .check(notNull())
                .checkEach(notNull(), When.isValid())
                .checkEach(validId(), When.isValid())
                .checkEach(inSet(unusedHyperGeoIds), When.isValid());
        return lvb.getResult();
    }
}
