package ru.yandex.direct.core.entity.idm.repository;

import java.util.List;
import java.util.Optional;

import org.jooq.Condition;
import org.jooq.DeleteConditionStep;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.idm.container.IdmGroupsMembersQueryFilter;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMember;
import ru.yandex.direct.dbschema.ppc.tables.records.IdmGroupsMembersRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.Tables.IDM_GROUPS_MEMBERS;
import static ru.yandex.direct.dbschema.ppc.Tables.INTERNAL_USERS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class IdmGroupsMembersRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqReaderWithSupplier<IdmGroupMember> jooqReader;
    private final JooqWriter<IdmGroupMember> jooqWriter;

    public IdmGroupsMembersRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqReader = createReader();
        jooqWriter = createWriter();
    }

    private static JooqReaderWithSupplier<IdmGroupMember> createReader() {
        return JooqReaderWithSupplierBuilder.builder(IdmGroupMember::new)
                .readProperty(IdmGroupMember.IDM_GROUP_ID, fromField(IDM_GROUPS_MEMBERS.IDM_GROUP_ID))
                .readProperty(IdmGroupMember.CLIENT_ID,
                        fromField(IDM_GROUPS_MEMBERS.CLIENT_ID).by(ClientId::fromLong))
                .readProperty(IdmGroupMember.LOGIN, fromField(USERS.LOGIN))
                .readProperty(IdmGroupMember.UID, fromField(USERS.UID))
                .readProperty(IdmGroupMember.DOMAIN_LOGIN, fromField(INTERNAL_USERS.DOMAIN_LOGIN))
                .build();
    }

    private JooqWriter<IdmGroupMember> createWriter() {
        return JooqWriterBuilder.<IdmGroupMember>builder()
                .writeField(IDM_GROUPS_MEMBERS.IDM_GROUP_ID, fromProperty(IdmGroupMember.IDM_GROUP_ID))
                .writeField(IDM_GROUPS_MEMBERS.CLIENT_ID,
                        fromProperty(IdmGroupMember.CLIENT_ID).by(ClientId::asLong))
                .build();
    }

    public List<IdmGroupMember> getMembers(Integer shard, IdmGroupsMembersQueryFilter filter) {
        Condition condition = DSL.trueCondition();
        if (filter.getLastMemberId() != null) {
            Long idmGroupId = filter.getLastMemberId().getIdmGroupId();
            Long clientId = filter.getLastMemberId().getClientId().asLong();
            Condition continueLastClient = IDM_GROUPS_MEMBERS.CLIENT_ID.eq(clientId)
                    .and(IDM_GROUPS_MEMBERS.IDM_GROUP_ID.greaterThan(idmGroupId));
            Condition continueNextClients = IDM_GROUPS_MEMBERS.CLIENT_ID.greaterThan(clientId);
            condition = condition.and(continueLastClient.or(continueNextClients));
        }
        return getMembers(shard, condition, filter.getLimitOffset());
    }

    public Optional<IdmGroupMember> getMember(int shard, ClientId clientId, Long idmGroupId) {
        Condition condition = IDM_GROUPS_MEMBERS.CLIENT_ID.eq(clientId.asLong())
                .and(IDM_GROUPS_MEMBERS.IDM_GROUP_ID.eq(idmGroupId));
        List<IdmGroupMember> members = getMembers(shard, condition, null);
        return members.stream().findFirst();
    }

    private List<IdmGroupMember> getMembers(Integer shard, Condition condition, LimitOffset limitOffset) {
        SelectConditionStep<Record> selectConditionStep = dslContextProvider.ppc(shard)
                .select(jooqReader.getFieldsToRead())
                .from(IDM_GROUPS_MEMBERS)
                .join(USERS).on(USERS.CLIENT_ID.eq(IDM_GROUPS_MEMBERS.CLIENT_ID))
                .join(INTERNAL_USERS).on(INTERNAL_USERS.UID.eq(USERS.UID))
                .where(condition);

        if (limitOffset != null) {
            selectConditionStep
                    .orderBy(IDM_GROUPS_MEMBERS.CLIENT_ID, IDM_GROUPS_MEMBERS.IDM_GROUP_ID)
                    .offset(limitOffset.offset())
                    .limit(limitOffset.limit());
        }

        return selectConditionStep
                .fetch(jooqReader::fromDb);
    }


    public void addMembersWhichNotExist(Integer shard, List<IdmGroupMember> members) {
        if (isEmpty(members)) {
            return;
        }
        InsertHelper<IdmGroupsMembersRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), IDM_GROUPS_MEMBERS)
                        .addAll(jooqWriter, members)
                        .onDuplicateKeyIgnore();
        insertHelper.executeIfRecordsAdded();
    }

    public void removeMembers(Integer shard, List<IdmGroupMember> members) {
        if (isEmpty(members)) {
            return;
        }
        List<Condition> orConditions = mapList(members, m -> IDM_GROUPS_MEMBERS.IDM_GROUP_ID.eq(m.getIdmGroupId())
                .and(IDM_GROUPS_MEMBERS.CLIENT_ID.eq(m.getClientId().asLong())));
        removeMembersByConditions(shard, orConditions);
    }

    public void removeMember(Integer shard, ClientId clientId, Long idmGroupId) {
        List<Condition> orConditions = singletonList(IDM_GROUPS_MEMBERS.IDM_GROUP_ID.eq(idmGroupId)
                .and(IDM_GROUPS_MEMBERS.CLIENT_ID.eq(clientId.asLong())));
        removeMembersByConditions(shard, orConditions);
    }

    private void removeMembersByConditions(Integer shard, List<Condition> orConditions) {
        DeleteConditionStep<IdmGroupsMembersRecord> step = dslContextProvider.ppc(shard)
                .deleteFrom(IDM_GROUPS_MEMBERS)
                .where(DSL.falseCondition());
        orConditions.forEach(step::or);
        step.execute();
    }

}
