package ru.yandex.direct.core.entity.idm.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.idm.model.IdmGroup;
import ru.yandex.direct.core.entity.idm.model.IdmRequiredRole;
import ru.yandex.direct.dbschema.ppcdict.enums.IdmGroupsRole;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static org.jooq.impl.DSL.trueCondition;
import static ru.yandex.direct.dbschema.ppcdict.tables.IdmGroups.IDM_GROUPS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class IdmGroupsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<IdmGroup> jooqMapper;

    public IdmGroupsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = createMapper();
    }

    private static JooqMapperWithSupplier<IdmGroup> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(IdmGroup::new)
                .map(property(IdmGroup.IDM_GROUP_ID, IDM_GROUPS.IDM_GROUP_ID))
                .map(convertibleProperty(IdmGroup.REQUIRED_ROLE, IDM_GROUPS.ROLE,
                        IdmGroupsRepository::idmRoleFromDb,
                        IdmGroupsRepository::idmRoleToDb))
                .build();
    }

    private static IdmRequiredRole idmRoleFromDb(IdmGroupsRole dbRole) {
        if (dbRole == IdmGroupsRole.manager) {
            return IdmRequiredRole.MANAGER;
        }
        throw new IllegalStateException("No such value: " + dbRole);
    }

    private static IdmGroupsRole idmRoleToDb(IdmRequiredRole coreRole) {
        if (coreRole == IdmRequiredRole.MANAGER) {
            return IdmGroupsRole.manager;
        }
        throw new IllegalStateException("No such value: " + coreRole);
    }

    public List<IdmGroup> getGroups(Collection<Long> groupIds) {
        if (isEmpty(groupIds)) {
            return emptyList();
        }
        Condition condition = IDM_GROUPS.IDM_GROUP_ID.in(groupIds);
        return getIdmGroups(condition);
    }

    public List<IdmGroup> getAllGroupsRoles() {
        return getIdmGroups(trueCondition());
    }

    private List<IdmGroup> getIdmGroups(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(IDM_GROUPS)
                .where(condition)
                .fetch(jooqMapper::fromDb);
    }

    public int add(Collection<IdmGroup> groups) {
        return new InsertHelper<>(dslContextProvider.ppcdict(), IDM_GROUPS)
                .addAll(jooqMapper, groups)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

}
