package ru.yandex.direct.core.entity.idm.repository;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.jooq.Condition;
import org.jooq.DeleteConditionStep;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.idm.container.IdmGroupsRoleQueryFilter;
import ru.yandex.direct.core.entity.idm.model.IdmGroupRole;
import ru.yandex.direct.dbschema.ppc.tables.records.IdmGroupRolesRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.Tables.IDM_GROUP_ROLES;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class IdmGroupsRolesRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqReaderWithSupplier<IdmGroupRole> jooqReader;
    private final JooqWriter<IdmGroupRole> jooqWriter;

    public IdmGroupsRolesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqReader = createReader();
        jooqWriter = createWriter();
    }

    private static JooqReaderWithSupplier<IdmGroupRole> createReader() {
        return JooqReaderWithSupplierBuilder.builder(IdmGroupRole::new)
                .readProperty(IdmGroupRole.IDM_GROUP_ID, fromField(IDM_GROUP_ROLES.IDM_GROUP_ID))
                .readProperty(IdmGroupRole.CLIENT_ID,
                        fromField(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID).by(ClientId::fromLong))
                .build();
    }

    private JooqWriter<IdmGroupRole> createWriter() {
        return JooqWriterBuilder.<IdmGroupRole>builder()
                .writeField(IDM_GROUP_ROLES.IDM_GROUP_ID, fromProperty(IdmGroupRole.IDM_GROUP_ID))
                .writeField(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID,
                        fromProperty(IdmGroupRole.CLIENT_ID).by(ClientId::asLong))
                .build();
    }

    public List<IdmGroupRole> getAllRoles(int shard) {
        return getRoles(shard, DSL.trueCondition(), null);
    }

    public Optional<IdmGroupRole> getRole(int shard, ClientId clientId, Long idmGroupId) {
        Condition condition = IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(clientId.asLong())
                .and(IDM_GROUP_ROLES.IDM_GROUP_ID.eq(idmGroupId));
        List<IdmGroupRole> roles = getRoles(shard, condition, null);
        return roles.stream().findFirst();
    }

    public List<IdmGroupRole> getRoles(int shard, IdmGroupsRoleQueryFilter filter) {
        Condition condition = DSL.trueCondition();
        if (filter.getLastReturnedRole() != null) {
            Long clientId = filter.getLastReturnedRole().getClientId().asLong();
            Long idmGroupId = filter.getLastReturnedRole().getIdmGroupId();
            Condition continueLastClient = IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(clientId)
                    .and(IDM_GROUP_ROLES.IDM_GROUP_ID.greaterThan(idmGroupId));
            Condition continueNextClients = IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.greaterThan(clientId);
            condition = condition.and(continueLastClient.or(continueNextClients));
        }
        return getRoles(shard, condition, filter.getLimitOffset());
    }

    /**
     * Возвращает групповые роли с доступом к клиенту {@code clientId}.
     * Не-bulk'овая, потому что используется только во внутренних отчётах
     */
    public List<IdmGroupRole> getRolesByClientId(int shard, ClientId clientId) {
        Condition condition = IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(clientId.asLong());
        return getRoles(shard, condition, null);
    }

    /**
     * Возвращает групповые роли по группе {@code idmGroupId}.
     * Не-bulk'овая, потому что используется только во внутренних отчётах
     */
    public List<IdmGroupRole> getRolesByIdmGroupId(int shard, Long idmGroupId) {
        Condition condition = IDM_GROUP_ROLES.IDM_GROUP_ID.eq(idmGroupId);
        return getRoles(shard, condition, null);
    }

    @QueryWithoutIndex("При пустом условии возможен выбор всех ролей подряд")
    private List<IdmGroupRole> getRoles(int shard, Condition condition, @Nullable LimitOffset limitOffset) {
        SelectConditionStep<Record> selectConditionStep = dslContextProvider.ppc(shard)
                .select(jooqReader.getFieldsToRead())
                .from(IDM_GROUP_ROLES)
                .where(condition);

        if (limitOffset != null) {
            selectConditionStep
                    .orderBy(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID, IDM_GROUP_ROLES.IDM_GROUP_ID)
                    .offset(limitOffset.offset())
                    .limit(limitOffset.limit());
        }

        return selectConditionStep
                .fetch(jooqReader::fromDb);
    }

    public void addRolesWhichNotExist(Integer shard, List<IdmGroupRole> roles) {
        if (isEmpty(roles)) {
            return;
        }
        InsertHelper<IdmGroupRolesRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), IDM_GROUP_ROLES)
                        .addAll(jooqWriter, roles)
                        .onDuplicateKeyIgnore();
        insertHelper.executeIfRecordsAdded();
    }

    public void removeRoles(Integer shard, List<IdmGroupRole> roles) {
        if (isEmpty(roles)) {
            return;
        }
        List<Condition> orConditions = mapList(roles, m -> IDM_GROUP_ROLES.IDM_GROUP_ID.eq(m.getIdmGroupId())
                .and(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(m.getClientId().asLong())));
        removeRolesByConditions(shard, orConditions);
    }

    public void removeRole(Integer shard, ClientId clientId, Long idmGroupId) {
        List<Condition> orConditions = singletonList(IDM_GROUP_ROLES.IDM_GROUP_ID.eq(idmGroupId)
                .and(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(clientId.asLong())));
        removeRolesByConditions(shard, orConditions);
    }

    private void removeRolesByConditions(Integer shard, List<Condition> orConditions) {
        DeleteConditionStep<IdmGroupRolesRecord> step = dslContextProvider.ppc(shard)
                .deleteFrom(IDM_GROUP_ROLES)
                .where(DSL.falseCondition());
        orConditions.forEach(step::or);
        step.execute();
    }

}
