package ru.yandex.direct.core.entity.idm.service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.idm.container.IdmGroupsMembersQueryFilter;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMember;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMemberId;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsMembersRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static ru.yandex.direct.core.entity.idm.container.IdmGroupsMembersQueryFilter.allIdmGroupsMembersFilter;
import static ru.yandex.direct.core.entity.idm.container.IdmGroupsMembersQueryFilter.getNextIdmGroupsMembersPageFilter;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Service
@ParametersAreNonnullByDefault
public class IdmGroupsMembersService {

    private final ShardHelper shardHelper;
    private final IdmGroupsMembersRepository idmGroupsMembersRepository;

    public IdmGroupsMembersService(ShardHelper shardHelper, IdmGroupsMembersRepository idmGroupsMembersRepository) {
        this.shardHelper = shardHelper;
        this.idmGroupsMembersRepository = idmGroupsMembersRepository;
    }

    public void addMembersWhichNotExist(List<IdmGroupMember> members) {
        shardHelper.groupByShard(members, ShardKey.CLIENT_ID, IdmGroupMember::getClientId)
                .getShardedDataMap()
                .forEach(idmGroupsMembersRepository::addMembersWhichNotExist);
    }

    public void removeMembers(List<IdmGroupMember> members) {
        shardHelper.groupByShard(members, ShardKey.CLIENT_ID, IdmGroupMember::getClientId)
                .getShardedDataMap()
                .forEach(idmGroupsMembersRepository::removeMembers);
    }

    public void removeMember(ClientId clientId, Long idmGroupId) {
        int shard = shardHelper.getShardByClientId(clientId);
        idmGroupsMembersRepository.removeMember(shard, clientId, idmGroupId);
    }

    public List<IdmGroupMember> getAllMembers() {
        return StreamEx.of(shardHelper.dbShards())
                .map(shard -> idmGroupsMembersRepository.getMembers(shard, allIdmGroupsMembersFilter()))
                .flatMap(StreamEx::of)
                .toList();
    }

    /**
     * Постранично возвращает всех членов всех IDM-групп.
     * Возвращаемые члены групп сортируются сначала по шарду, потом по ClientId, потом по GroupId. Если lastMemberId
     * равен null, то возвращаются члены с начала первого шарда. Если lastMemberId не равен null, то находится шард на
     * котором он хранится и выбираются члены групп начиная со следующего. Если в текущем шарде не хватило данных для
     * заполнения страницы размером pageSize, то выборка переходит на начало следующего шарда, и так до заполнения
     * страницы или окончания перебора всех шардов.
     */
    public List<IdmGroupMember> getNextMembersPage(@Nullable IdmGroupMemberId lastMemberId, int pageSize) {
        int continueShard = nvl(ifNotNull(lastMemberId, m -> shardHelper.getShardByClientId(m.getClientId())), 0);
        Iterator<Integer> shards = StreamEx.of(shardHelper.dbShards())
                .filter(shard -> continueShard <= shard)
                .sorted()
                .iterator();
        ArrayList<IdmGroupMember> members = new ArrayList<>(pageSize);
        int limit = pageSize - members.size();
        while (shards.hasNext() && 0 < limit) {
            IdmGroupsMembersQueryFilter filter = getNextIdmGroupsMembersPageFilter(lastMemberId, limit);
            List<IdmGroupMember> shardGroupRoles = idmGroupsMembersRepository.getMembers(shards.next(), filter);
            members.addAll(shardGroupRoles);
            lastMemberId = null;
            limit = pageSize - members.size();
        }
        return members;
    }

    public Optional<IdmGroupMember> getMember(ClientId clientId, Long idmGroupId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return idmGroupsMembersRepository.getMember(shard, clientId, idmGroupId);
    }

}
