package ru.yandex.direct.core.entity.idm.service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.idm.container.IdmGroupsRoleQueryFilter;
import ru.yandex.direct.core.entity.idm.model.IdmGroupRole;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRolesRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Service
@ParametersAreNonnullByDefault
public class IdmGroupsRolesService {

    private final ShardHelper shardHelper;
    private final IdmGroupsRolesRepository idmGroupsRolesRepository;

    public IdmGroupsRolesService(ShardHelper shardHelper, IdmGroupsRolesRepository idmGroupsRolesRepository) {
        this.shardHelper = shardHelper;
        this.idmGroupsRolesRepository = idmGroupsRolesRepository;
    }

    public void addRolesWhichNotExist(List<IdmGroupRole> roles) {
        shardHelper.groupByShard(roles, ShardKey.CLIENT_ID, IdmGroupRole::getClientId)
                .getShardedDataMap()
                .forEach(idmGroupsRolesRepository::addRolesWhichNotExist);
    }

    public void removeRoles(List<IdmGroupRole> roles) {
        shardHelper.groupByShard(roles, ShardKey.CLIENT_ID, IdmGroupRole::getClientId)
                .getShardedDataMap()
                .forEach(idmGroupsRolesRepository::removeRoles);
    }

    public void removeRole(ClientId clientId, Long idmGroupId) {
        int shard = shardHelper.getShardByClientId(clientId);
        idmGroupsRolesRepository.removeRole(shard, clientId, idmGroupId);
    }

    public List<IdmGroupRole> getAllRoles() {
        return StreamEx.of(shardHelper.dbShards())
                .map(idmGroupsRolesRepository::getAllRoles)
                .flatMap(StreamEx::of)
                .toList();
    }

    /**
     * Возвращает групповые роли с доступом к клиенту {@code clientId}.
     * Не-bulk'овая, потому что используется только во внутренних отчётах
     */
    public List<IdmGroupRole> getRolesByClientId(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return idmGroupsRolesRepository.getRolesByClientId(shard, clientId);
    }

    /**
     * Возвращает групповые роли по группе {@code idmGroupId}.
     * Не-bulk'овая, потому что используется только во внутренних отчётах
     */
    public List<IdmGroupRole> getRolesByGroupId(Long groupId) {
        var result = new ArrayList<IdmGroupRole>();
        shardHelper.forEachShard(shard -> result.addAll(idmGroupsRolesRepository.getRolesByIdmGroupId(shard, groupId)));
        return result;
    }

    /**
     * Возвращает набор групповых ролей заданной длинны начиная начиная со следующей после последней возвращённой.
     * Набор отсортирован сначала по шарду, потом по SubjectClientId, потом по IdmGroupId. Если последняя возвращённая
     * роль не задана, то  возвращаются роли с начала первого шарда.
     */
    public List<IdmGroupRole> getNextPageGroupRoles(@Nullable IdmGroupRole lastReturnedRole, int pageSize) {
        int continueShard = nvl(ifNotNull(lastReturnedRole, m -> shardHelper.getShardByClientId(m.getClientId())), 0);
        Iterator<Integer> shards = StreamEx.of(shardHelper.dbShards())
                .filter(shard -> continueShard <= shard)
                .sorted()
                .iterator();
        ArrayList<IdmGroupRole> roles = new ArrayList<>(pageSize);
        int limit = pageSize - roles.size();
        while (shards.hasNext() && 0 < limit) {
            IdmGroupsRoleQueryFilter filter = IdmGroupsRoleQueryFilter.getIdmGroupRolesPage(lastReturnedRole, limit);
            List<IdmGroupRole> shardGroupRoles = idmGroupsRolesRepository.getRoles(shards.next(), filter);
            roles.addAll(shardGroupRoles);
            lastReturnedRole = null;
            limit = pageSize - roles.size();
        }
        return roles;
    }

    public Optional<IdmGroupRole> getRole(ClientId clientId, Long idmGroupId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return idmGroupsRolesRepository.getRole(shard, clientId, idmGroupId);
    }

}
