package ru.yandex.direct.core.entity.image.converter;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang.StringUtils;

import ru.yandex.direct.avatars.client.model.AvatarInfo;
import ru.yandex.direct.avatars.client.model.answer.ImageSize;
import ru.yandex.direct.avatars.client.model.answer.SmartArea;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.model.ImageFormat;
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta;
import ru.yandex.direct.core.entity.image.model.ImageSizeMeta;
import ru.yandex.direct.core.entity.image.model.ImageSmartCenter;
import ru.yandex.direct.core.entity.image.service.ImageUtils;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesFormatsAvatarsHost;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_NAME_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_FORMATS_BY_IMAGE_TYPE;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.calculateImageTypeOfLogoExtendedImage;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.calculateImageTypeOfTextBannerImage;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
public class BannerImageConverter {

    public static BannerImageFromPool toBannerImageFromPoolInformation(long clientId, String name, String hash,
                                                                       BannerImageSource source) {
        return new BannerImageFromPool()
                .withImageHash(hash)
                .withCreateTime(LocalDateTime.now())
                .withName(StringUtils.substring(name, 0, MAX_IMAGE_NAME_SIZE))
                .withSource(source)
                .withClientId(clientId);
    }

    public static BannerImageFormat toBannerImageFormat(ru.yandex.direct.core.entity.banner.model.ImageSize imageSize,
                                                        BannerImageType bannerImageType, String avatarHost,
                                                        String imageHash, AvatarInfo avatarInfo) {
        ImageType imageType = toImageType(imageSize, bannerImageType);
        return new BannerImageFormat()
                .withImageHash(imageHash)
                .withImageType(imageType)
                .withAvatarsHost(toAvatarHost(avatarHost))
                .withFormats(toSupportedImageFormats(avatarInfo.getSizes(), imageType))
                .withSize(imageSize)
                .withMdsMeta(avatarInfo.getMeta())
                .withMdsGroupId(avatarInfo.getGroupId())
                .withNamespace(toBannerImageFormatNamespace(avatarInfo.getNamespace()));
    }

    public static ImageType toImageType(ru.yandex.direct.core.entity.banner.model.ImageSize imageSize,
                                        BannerImageType bannerImageType) {
        if (bannerImageType == BannerImageType.BANNER_TEXT || bannerImageType == BannerImageType.ASSET_MULTICARD ||
            bannerImageType == BannerImageType.OFFER_IMAGE) {
            return calculateImageTypeOfTextBannerImage(imageSize);
        } else if (bannerImageType == BannerImageType.ASSET_LOGO_EXTENDED) {
            return calculateImageTypeOfLogoExtendedImage(imageSize);
        } else if (bannerImageType == BannerImageType.ASSET_LOGO) {
            return ImageType.LOGO;
        } else {
            return ImageType.IMAGE_AD;
        }
    }

    @Nullable
    public static ImageMdsMeta toImageMdsMeta(@Nullable String json) {
        return ifNotNull(json, j -> JsonUtils.fromJson(j, ImageMdsMeta.class));
    }

    @Nullable
    public static ImageMdsMeta mergeImageMdsMeta(@Nullable ImageMdsMeta mdsMeta,
                                                 @Nullable ImageMdsMeta overridingMdsMeta) {
        if (mdsMeta == null || overridingMdsMeta == null) {
            return mdsMeta;
        }
        ImageMdsMeta mergedMdsMeta = ImageUtils.copyImageMdsMeta(mdsMeta);
        Map<String, ImageSizeMeta> sizes = overridingMdsMeta.getSizes();
        for (String size : sizes.keySet()) {
            ImageSizeMeta sizeMeta = sizes.get(size);
            Map<String, ImageSmartCenter> smartCenters = sizeMeta.getSmartCenters();
            for (String ratio : smartCenters.keySet()) {
                mergedMdsMeta.getSizes().get(size).getSmartCenters().put(ratio, smartCenters.get(ratio));
            }
        }
        return mergedMdsMeta;
    }

    @Nullable
    public static BannerImageFormatNamespace toBannerImageFormatNamespace(@Nullable String namespace) {
        if (namespace == null) {
            return null;
        }
        switch (namespace) {
            case "direct":
                return BannerImageFormatNamespace.DIRECT;
            case "direct-picture":
                return BannerImageFormatNamespace.DIRECT_PICTURE;
            default:
                throw new IllegalStateException("No such value: " + namespace);
        }
    }

    public static Map<String, ImageFormat> toSupportedImageFormats(Map<String, ImageSize> imageSizes,
                                                                   ImageType imageType) {
        return EntryStream.of(imageSizes)
                .filterKeys(SUPPORTED_FORMATS_BY_IMAGE_TYPE.get(imageType)::contains)
                .mapValues(imageSize -> BannerImageConverter.toImageFormat(imageSize, imageType))
                .toMap();
    }

    private static ImageFormat toImageFormat(ImageSize imageSize, ImageType imageType) {
        ImageFormat imageFormat = new ImageFormat()
                .withHeight(imageSize.getHeight())
                .withWidth(imageSize.getWidth())
                .withSmartCenter(toImageSmartCenter(imageSize.getSmartCenter()));
        if (imageType == ImageType.IMAGE_AD) {
            imageFormat.setPath(imageSize.getPath());
        }
        return imageFormat;
    }

    private static ImageSmartCenter toImageSmartCenter(@Nullable SmartArea smartCenter) {
        return ifNotNull(smartCenter, sc -> new ImageSmartCenter()
                .withHeight(sc.getHeight())
                .withWidth(sc.getWidth())
                .withX(sc.getX())
                .withY(sc.getY()));
    }

    private static AvatarHost toAvatarHost(String host) {
        if (BannerImagesFormatsAvatarsHost.avatars_mds_yandex_net.getLiteral().equals(host)) {
            return AvatarHost.AVATARS_MDS_YANDEX_NET;
        }

        if (BannerImagesFormatsAvatarsHost.avatars_mdst_yandex_net.getLiteral().equals(host)) {
            return AvatarHost.AVATARS_MDST_YANDEX_NET;
        }

        throw new IllegalArgumentException("unsupported avatar host");
    }

}
