package ru.yandex.direct.core.entity.image.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.image.repository.mapper.BannerImageMapperProvider.getBannerImageFormatMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesFormats.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesPool.BANNER_IMAGES_POOL;

@Repository
@ParametersAreNonnullByDefault
public class BannerImageFormatRepository {
    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<BannerImageFormat> bannerImageFormatMapper;

    @Autowired
    public BannerImageFormatRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        bannerImageFormatMapper = getBannerImageFormatMapper();
    }

    /**
     * Получает информация об изображении прикрепленном к баннеру, полученную ранее из аватарницы
     */
    public Map<String, BannerImageFormat> getBannerImageFormats(int shard, Collection<String> imageHashes) {
        if (imageHashes.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(bannerImageFormatMapper.getFieldsToRead())
                .from(BANNER_IMAGES_FORMATS)
                .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.in(imageHashes))
                .fetchMap(BANNER_IMAGES_FORMATS.IMAGE_HASH, bannerImageFormatMapper::fromDb);
    }


    /**
     * Получает информация об изображении прикрепленном к баннеру, полученную ранее из аватарницы
     */
    public Map<String, BannerImageFormat> getBannerImageFormats(int shard, ClientId clientId,
                                                                Collection<String> imageHashes) {
        if (imageHashes.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(bannerImageFormatMapper.getFieldsToRead())
                .from(BANNER_IMAGES_FORMATS)
                .join(BANNER_IMAGES_POOL).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.in(imageHashes))
                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong()))
                .fetchMap(BANNER_IMAGES_FORMATS.IMAGE_HASH, bannerImageFormatMapper::fromDb);
    }


    /**
     * Получение сохраненных в БД хешей изображений и типов принадлежащих указанному клиенту из переданной коллекции
     *
     * @param shard       шард
     * @param clientId    id клиента
     * @param imageHashes коллекция проверяемых хешей
     * @return коллекция принадлежащих указанному клиенту хешей изображений и типов из переданной коллекции
     */
    public Map<String, ImageType> getExistingImageHashesWithType(int shard, ClientId clientId,
                                                                 Collection<String> imageHashes) {
        if (imageHashes.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(asList(BANNER_IMAGES_FORMATS.IMAGE_HASH, BANNER_IMAGES_FORMATS.IMAGE_TYPE))
                .from(BANNER_IMAGES_FORMATS)
                .join(BANNER_IMAGES_POOL).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.in(imageHashes))
                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong()))
                .fetchMap(BANNER_IMAGES_FORMATS.IMAGE_HASH,
                        r -> ImageType.fromSource(r.getValue(BANNER_IMAGES_FORMATS.IMAGE_TYPE)));
    }

    /**
     * Получение сохраненных в БД размеров принадлежащих указанному клиенту из переданной коллекции
     *
     * @param shard    шард
     * @param clientId id клиента
     * @param hashes   коллекция проверяемых хешей
     * @return коллекция принадлежащих указанному клиенту размеров из переданной коллекции
     */
    public Map<String, Pair<ImageType, ImageSize>> getExistingImageSizes(int shard, ClientId clientId,
                                                                         Collection<String> hashes) {
        if (hashes.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES_POOL.IMAGE_HASH, BANNER_IMAGES_FORMATS.IMAGE_TYPE,
                        BANNER_IMAGES_FORMATS.WIDTH, BANNER_IMAGES_FORMATS.HEIGHT)
                .from(BANNER_IMAGES_POOL)
                .join(BANNER_IMAGES_FORMATS).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES_POOL.IMAGE_HASH.in(hashes))
                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong()))
                .fetchMap(BANNER_IMAGES_POOL.IMAGE_HASH, r -> Pair.of(
                        ImageType.fromSource(r.get(BANNER_IMAGES_FORMATS.IMAGE_TYPE)),
                        new ImageSize()
                                .withWidth(r.get(BANNER_IMAGES_FORMATS.WIDTH).intValue())
                                .withHeight(r.get(BANNER_IMAGES_FORMATS.HEIGHT).intValue())));
    }

    public void addBannerImageFormat(int shard, List<BannerImageFormat> bannerImageFormats) {
        for (var format : bannerImageFormats) {
            checkState(!format.getFormats().isEmpty(), "empty formats");
        }
        new InsertHelper<>(dslContextProvider.ppc(shard), BANNER_IMAGES_FORMATS)
                .addAll(bannerImageFormatMapper, bannerImageFormats)
                .executeIfRecordsAdded();
    }

    public void updateBannerImageFormatSize(int shard, BannerImageFormat bannerImageFormat) {
            checkState(!bannerImageFormat.getFormats().isEmpty(), "empty formats");

            dslContextProvider.ppc(shard)
                    .update(BANNER_IMAGES_FORMATS)
                    .set(BANNER_IMAGES_FORMATS.HEIGHT, bannerImageFormat.getSize().getHeight().longValue())
                    .set(BANNER_IMAGES_FORMATS.WIDTH, bannerImageFormat.getSize().getWidth().longValue())
                    .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(bannerImageFormat.getImageHash()))
                    .execute();
    }

    /**
     * Возвращает список image_hash элементов у которых нет связи с таблицами BANNER_IMAGES и BANNER_IMAGES_POOL (по
     * IMAGE_HASH) и c лимитом {@param limit}
     *
     * @param shard шард
     * @param limit количество строк
     */
    public List<String> getHashesForDelete(int shard, int limit) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES_FORMATS.IMAGE_HASH)
                .from(BANNER_IMAGES_FORMATS)
                .leftJoin(BANNER_IMAGES).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES.IMAGE_HASH))
                .leftJoin(BANNER_IMAGES_POOL).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES.IMAGE_HASH.isNull())
                .and(Tables.BANNER_IMAGES_POOL.IMAGE_HASH.isNull())
                .limit(limit)
                .fetch(BANNER_IMAGES_FORMATS.IMAGE_HASH);
    }

    /**
     * Удаляет строки из banner_images_formats по IMAGE_HASH {@param hashes} и у которых нет зависимости с таблицами
     * BANNER_IMAGES и BANNER_IMAGES_POOL (по IMAGE_HASH)
     *
     * @param shard  шард
     * @param hashes коллекция image_hash
     * @return количество удаленных строк
     */
    public int deleteByHashesWithoutRelations(int shard, Collection<String> hashes) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(BANNER_IMAGES_FORMATS)
                .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.in(hashes))
                .andNotExists(dslContextProvider.ppc(shard)
                        .selectOne()
                        .from(BANNER_IMAGES)
                        .where(BANNER_IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH)))
                .andNotExists(dslContextProvider.ppc(shard)
                        .selectOne()
                        .from(Tables.BANNER_IMAGES_POOL)
                        .where(Tables.BANNER_IMAGES_POOL.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH)))
                .execute();
    }

    /**
     * По хэшам изображений получает id баннеров, к которым эти изображения привязаны
     *
     * @param imageHashes коллекция хэшей изображений, для которых ищем привязку к баннеру
     * @return набор id баннеров
     */
    public Map<String, List<Long>> getBannerIdsByImageHashes(int shard, Collection<String> imageHashes) {
        if (imageHashes.isEmpty()) {
            return Map.of();
        }

        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES.IMAGE_HASH, BANNER_IMAGES.BID)
                .from(BANNER_IMAGES)
                .where(BANNER_IMAGES.IMAGE_HASH.in(imageHashes))
                .fetchGroups(BANNER_IMAGES.IMAGE_HASH, BANNER_IMAGES.BID);
    }
}
