package ru.yandex.direct.core.entity.image.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.image.repository.mapper.BannerImageMapperProvider.getBannerImageFromPoolMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_POOL;

@Repository
@ParametersAreNonnullByDefault
public class BannerImagePoolRepository {
    private final JooqMapperWithSupplier<BannerImageFromPool> mapper;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    public BannerImagePoolRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.mapper = getBannerImageFromPoolMapper();
    }

    /**
     * Добавляет клиенту изображения в его пул изображений
     *
     * @param shard    шард
     * @param clientId id клиента, которому добавляем изображения
     * @param images   информация об изображениях
     */
    public void addOrUpdateImagesToPool(int shard, ClientId clientId, List<BannerImageFromPool> images) {
        generateIdsForBannerImageFromPool(clientId, images);
        new InsertHelper<>(dslContextProvider.ppc(shard), BANNER_IMAGES_POOL)
                .addAll(mapper, images)
                .onDuplicateKeyUpdate()
                .set(BANNER_IMAGES_POOL.NAME, MySQLDSL.values(BANNER_IMAGES_POOL.NAME))
                .execute();
    }

    private void generateIdsForBannerImageFromPool(ClientId clientId, List<BannerImageFromPool> images) {
        List<Long> ids = shardHelper.generateBannerPoolImageIds(clientId.asLong(), images.size());
        StreamEx.of(images).zipWith(ids.stream())
                .forKeyValue(BannerImageFromPool::setId);
    }

    public Map<String, BannerImageFromPool> getBannerImageFromPoolsByHashes(int shard, ClientId clientId,
                                                                            Collection<String> imageHashes) {
        if (imageHashes.isEmpty()) {
            return Collections.emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(BANNER_IMAGES_POOL)
                .where(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong())
                        .and(BANNER_IMAGES_POOL.IMAGE_HASH.in(imageHashes)))
                .fetchMap(BANNER_IMAGES_POOL.IMAGE_HASH, mapper::fromDb);
    }
}
