package ru.yandex.direct.core.entity.image.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.feature.service.FeatureHelper;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.core.entity.image.container.ImageFilterContainer;
import ru.yandex.direct.core.entity.image.converter.BannerImageConverter;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.model.ImagePoolKey;
import ru.yandex.direct.core.entity.image.service.ImageConstants;
import ru.yandex.direct.core.entity.image.service.validation.type.ImageSaveValidationForOfferImageSupport;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesFormatsImageType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;

import static com.google.common.collect.Iterables.getFirst;
import static java.util.function.Function.identity;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.core.entity.image.repository.mapper.BannerImageMapperProvider.getImageMapper;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.LOGO_EXTENDED_IMAGE_FORMAT_TYPES;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_LOGO_EXTENDED_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MIN_LOGO_EXTENDED_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MIN_LOGO_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.TEXT_BANNER_IMAGE_FORMAT_TYPES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesFormats.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesPool.BANNER_IMAGES_POOL;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ImageDataRepository {
    private final static Condition IS_SIZE_AVAILABLE_FOR_LOGO =
            BANNER_IMAGES_FORMATS.WIDTH.eq(BANNER_IMAGES_FORMATS.HEIGHT)
                    .and(BANNER_IMAGES_FORMATS.WIDTH.ge(Long.valueOf(MIN_LOGO_SIZE)));

    private final static Condition IS_SIZE_AVAILABLE_FOR_LOGO_EXTENDED =
            BANNER_IMAGES_FORMATS.WIDTH.between(Long.valueOf(MIN_LOGO_EXTENDED_SIZE), Long.valueOf(MAX_LOGO_EXTENDED_SIZE))
                    .and(BANNER_IMAGES_FORMATS.HEIGHT.between(Long.valueOf(MIN_LOGO_EXTENDED_SIZE),
                            Long.valueOf(MAX_LOGO_EXTENDED_SIZE)));

    private final JooqMapperWithSupplier<Image> mapper;
    private final DslContextProvider dslContextProvider;
    private final Condition mcbannerSizeCondition;
    private final Condition imageAdSizeCondition;
    private final Condition imageAdSizeOriginalCondition;

    public ImageDataRepository(DslContextProvider dslContextProvider) {
        this.mapper = getImageMapper();
        this.dslContextProvider = dslContextProvider;

        mcbannerSizeCondition = StreamEx.of(ImageConstants.ALLOWED_SIZES_FOR_MCBANNER)
                .map(this::sizeEqualCondition)
                .foldLeft(Condition::or)
                .get();

        imageAdSizeCondition = StreamEx.of(ImageConstants.ALLOWED_SIZES_FOR_AD_IMAGE)
                .map(this::sizeEqualCondition)
                .foldLeft(Condition::or)
                .get();

        imageAdSizeOriginalCondition = StreamEx.of(ImageConstants.ALLOWED_SIZES_FOR_AD_IMAGE_ORIGINAL)
                .map(this::sizeEqualCondition)
                .foldLeft(Condition::or)
                .get();
    }

    public void overrideImageSmartCenters(int shard, ClientId clientId, String imageHash, String smartCenters) {
        dslContextProvider.ppc(shard)
                .update(BANNER_IMAGES_POOL)
                .set(BANNER_IMAGES_POOL.MDS_META_USER_OVERRIDE, smartCenters)
                .where(BANNER_IMAGES_POOL.IMAGE_HASH.eq(imageHash))
                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }

    private Condition sizeEqualCondition(ImageSize imageSize) {
        return BANNER_IMAGES_FORMATS.HEIGHT.eq(imageSize.getHeight().longValue())
                .and(BANNER_IMAGES_FORMATS.WIDTH.eq(imageSize.getWidth().longValue()));
    }

    public Image getImage(int shard, ClientId clientId, String imageHash) {
        ImageFilterContainer filter = new ImageFilterContainer().withImageHashes(Set.of(imageHash));
        List<Image> images = getImages(shard, clientId, filter);
        return getFirst(images, null);
    }

    public List<Image> getImages(int shard, ClientId clientId, ImageFilterContainer filterContainer) {
        SelectConditionStep<Record> conditionStep = dslContextProvider.ppc(shard).select(mapper.getFieldsToRead())
                .from(BANNER_IMAGES_POOL)
                .innerJoin(BANNER_IMAGES_FORMATS)
                .on(BANNER_IMAGES_POOL.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                .where(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong()));

        if (CollectionUtils.isNotEmpty(filterContainer.getImageHashes())) {
            conditionStep.and(BANNER_IMAGES_POOL.IMAGE_HASH.in(filterContainer.getImageHashes()));
        }

        if (filterContainer.getBannerImageType() != null) {
            filterByBannerType(filterContainer, conditionStep);
        }

        if (filterContainer.getMinCreateTime() != null) {
            conditionStep.and(BANNER_IMAGES_POOL.CREATE_TIME.ge(filterContainer.getMinCreateTime()));
        }

        if (filterContainer.getMaxCreateTime() != null) {
            conditionStep.and(BANNER_IMAGES_POOL.CREATE_TIME.le(filterContainer.getMaxCreateTime()));
        }

        if (filterContainer.getWidth() != null) {
            conditionStep.and(BANNER_IMAGES_FORMATS.WIDTH.eq(Long.valueOf(filterContainer.getWidth())));
        }

        if (filterContainer.getHeight() != null) {
            conditionStep.and(BANNER_IMAGES_FORMATS.HEIGHT.eq(Long.valueOf(filterContainer.getHeight())));
        }

        if (filterContainer.getMaxCreateTime() != null) {
            conditionStep.and(BANNER_IMAGES_POOL.CREATE_TIME.le(filterContainer.getMaxCreateTime()));
        }

        if (isNotBlank(filterContainer.getNameContains())) {
            conditionStep.and(BANNER_IMAGES_POOL.NAME.contains(filterContainer.getNameContains()));
        }

        if (filterContainer.getSource() != null) {
            conditionStep.and(BANNER_IMAGES_POOL.SOURCE.eq(BannerImageSource.toSource(filterContainer.getSource())));
        }

        List<Image> images = conditionStep.fetch(mapper::fromDb);

        return StreamEx.of(images)
                .map(image -> image.withMdsMetaWithUserSettings(
                        BannerImageConverter.mergeImageMdsMeta(image.getMdsMeta(), image.getMdsMetaUserOverride())))
                .filter(image -> isAllowedFormatIfOfferImage(image, filterContainer.getBannerImageType()))
                .toList();
    }

    private void filterByBannerType(ImageFilterContainer filterContainer, SelectConditionStep<Record> conditionStep) {
        switch (filterContainer.getBannerImageType()) {
            case BANNER_TEXT:
            case ASSET_MULTICARD:
            case OFFER_IMAGE:
                conditionStep.and(BANNER_IMAGES_FORMATS.IMAGE_TYPE
                        .in(TEXT_BANNER_IMAGE_FORMAT_TYPES));
                break;
            case BANNER_IMAGE_AD:
                var imageSizeCondition = FeatureHelper.feature(FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES).enabled()
                        ? imageAdSizeOriginalCondition
                        : imageAdSizeCondition;
                conditionStep.and(imageSizeCondition)
                        .and(BANNER_IMAGES_FORMATS.IMAGE_TYPE.eq(BannerImagesFormatsImageType.image_ad));
                break;
            case BANNER_MCBANNER:
                conditionStep.and(mcbannerSizeCondition)
                        .and(BANNER_IMAGES_FORMATS.IMAGE_TYPE.eq(BannerImagesFormatsImageType.image_ad));
                break;
            case BANNER_INTERNAL:
                conditionStep.and(BANNER_IMAGES_FORMATS.IMAGE_TYPE.eq(BannerImagesFormatsImageType.image_ad));
                break;
            case ASSET_LOGO:
                conditionStep.and(BANNER_IMAGES_FORMATS.IMAGE_TYPE.eq(BannerImagesFormatsImageType.logo))
                        .and(IS_SIZE_AVAILABLE_FOR_LOGO);
                break;
            case ASSET_LOGO_EXTENDED:
                conditionStep.and(BANNER_IMAGES_FORMATS.IMAGE_TYPE.in(LOGO_EXTENDED_IMAGE_FORMAT_TYPES))
                        .and(IS_SIZE_AVAILABLE_FOR_LOGO_EXTENDED);
                break;
            default:
                throw new IllegalArgumentException(String.format("Images are not supported for banner type: %s",
                        filterContainer.getBannerImageType()));
        }
    }

    private Boolean isAllowedFormatIfOfferImage(Image image, BannerImageType imageType) {
        if (imageType != BannerImageType.OFFER_IMAGE) {
            return true;
        }

        ImageFileFormat fileFormat = ImageFileFormat.valueOf(image.getMdsMeta().getMeta().getOrigFormat());

        return ImageSaveValidationForOfferImageSupport.ALLOWED_FORMATS.contains(fileFormat);
    }

    public Map<ImagePoolKey, Image> getImages(int shard, Collection<ImagePoolKey> imageKeys) {
        var filterRows =
                mapList(imageKeys, key -> row(key.getClientId().asLong(), key.getImageHash()));

        List<Image> images = dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(BANNER_IMAGES_POOL)
                .innerJoin(BANNER_IMAGES_FORMATS)
                .on(BANNER_IMAGES_POOL.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                .where(row(BANNER_IMAGES_POOL.CLIENT_ID, BANNER_IMAGES_POOL.IMAGE_HASH).in(filterRows))
                .fetch(mapper::fromDb);

        return StreamEx.of(images)
                .mapToEntry(image ->
                        new ImagePoolKey(ClientId.fromLong(image.getClientId()), image.getImageHash()), identity())
                .mapValues(image -> image.withMdsMetaWithUserSettings(
                        BannerImageConverter.mergeImageMdsMeta(image.getMdsMeta(), image.getMdsMetaUserOverride())))
                .toMap();
    }
}
