package ru.yandex.direct.core.entity.image.repository.mapper;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.image.converter.BannerImageConverter;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.model.ImageFormat;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_POOL;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesFormats.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@ParametersAreNonnullByDefault
public class BannerImageMapperProvider {
    private static final JooqMapperWithSupplier<BannerImageFormat> BANNER_IMAGE_FORMAT_MAPPER =
            createBannerImageFormatMapper();
    private static final JooqMapperWithSupplier<BannerImageFromPool> BANNER_IMAGE_FROM_POOL_MAPPER =
            createBannerImageFromPoolMapper();
    private static final JooqMapperWithSupplier<Image> IMAGE_MAPPER = createImageMapper();

    private BannerImageMapperProvider() {
    }


    public static JooqMapperWithSupplier<BannerImageFormat> getBannerImageFormatMapper() {
        return BANNER_IMAGE_FORMAT_MAPPER;
    }

    public static JooqMapperWithSupplier<BannerImageFromPool> getBannerImageFromPoolMapper() {
        return BANNER_IMAGE_FROM_POOL_MAPPER;
    }

    public static JooqMapperWithSupplier<Image> getImageMapper() {
        return IMAGE_MAPPER;
    }

    private static JooqMapperWithSupplier<BannerImageFormat> createBannerImageFormatMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerImageFormat::new)
                .map(property(BannerImageFormat.IMAGE_HASH, BANNER_IMAGES_FORMATS.IMAGE_HASH))
                .map(integerProperty(BannerImageFormat.MDS_GROUP_ID, BANNER_IMAGES_FORMATS.MDS_GROUP_ID))
                .map(convertibleProperty(BannerImageFormat.NAMESPACE, BANNER_IMAGES_FORMATS.NAMESPACE,
                        BannerImageFormatNamespace::fromSource, BannerImageFormatNamespace::toSource))
                .map(convertibleProperty(BannerImageFormat.IMAGE_TYPE, BANNER_IMAGES_FORMATS.IMAGE_TYPE,
                        ImageType::fromSource, ImageType::toSource))
                .readProperty(BannerImageFormat.SIZE,
                        fromFields(BANNER_IMAGES_FORMATS.WIDTH, BANNER_IMAGES_FORMATS.HEIGHT)
                                .by(BannerImageMapperProvider::imageSizeFromDb))
                .writeField(BANNER_IMAGES_FORMATS.WIDTH,
                        fromProperty(BannerImageFormat.SIZE).by(BannerImageMapperProvider::dbWidthFromImageSize))
                .writeField(BANNER_IMAGES_FORMATS.HEIGHT,
                        fromProperty(BannerImageFormat.SIZE).by(BannerImageMapperProvider::dbHeightFromImageSize))
                .map(convertibleProperty(BannerImageFormat.FORMATS, BANNER_IMAGES_FORMATS.FORMATS,
                        BannerImageMapperProvider::toImageFormats, JsonUtils::toJsonNullable))
                .map(convertibleProperty(BannerImageFormat.AVATARS_HOST, BANNER_IMAGES_FORMATS.AVATARS_HOST,
                        AvatarHost::fromSource, AvatarHost::toSource))
                .map(property(BannerImageFormat.MDS_META, BANNER_IMAGES_FORMATS.MDS_META))
                .build();
    }

    private static ImageSize imageSizeFromDb(Long width, Long height) {
        return new ImageSize()
                .withWidth(width.intValue())
                .withHeight(height.intValue());
    }

    private static long dbWidthFromImageSize(ImageSize imageSize) {
        return imageSize.getWidth().longValue();
    }

    private static long dbHeightFromImageSize(ImageSize imageSize) {
        return imageSize.getHeight().longValue();
    }

    private static Map<String, ImageFormat> toImageFormats(String textImageFormats) {
        TypeReference<Map<String, ImageFormat>> typeRef = new TypeReference<Map<String, ImageFormat>>() {
        };
        return JsonUtils.fromJson(textImageFormats, typeRef);
    }

    private static JooqMapperWithSupplier<BannerImageFromPool> createBannerImageFromPoolMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerImageFromPool::new)
                .map(property(BannerImageFromPool.ID, BANNER_IMAGES_POOL.IMP_ID))
                .map(property(BannerImageFromPool.CLIENT_ID, BANNER_IMAGES_POOL.CLIENT_ID))
                .map(property(BannerImageFromPool.NAME, BANNER_IMAGES_POOL.NAME))
                .map(property(BannerImageFromPool.IMAGE_HASH, BANNER_IMAGES_POOL.IMAGE_HASH))
                .map(property(BannerImageFromPool.CREATE_TIME, BANNER_IMAGES_POOL.CREATE_TIME))
                .map(convertibleProperty(BannerImageFromPool.SOURCE, BANNER_IMAGES_POOL.SOURCE,
                        BannerImageSource::fromSource,
                        BannerImageSource::toSource))
                .build();
    }

    private static JooqMapperWithSupplier<Image> createImageMapper() {
        return JooqMapperWithSupplierBuilder.builder(Image::new)
                .map(property(Image.CLIENT_ID, BANNER_IMAGES_POOL.CLIENT_ID))
                .map(property(Image.IMAGE_HASH, BANNER_IMAGES_POOL.IMAGE_HASH))
                .map(property(Image.NAME, BANNER_IMAGES_POOL.NAME))
                .map(property(Image.CREATE_TIME, BANNER_IMAGES_POOL.CREATE_TIME))
                .map(integerProperty(Image.MDS_GROUP_ID, BANNER_IMAGES_FORMATS.MDS_GROUP_ID))
                .map(convertibleProperty(Image.NAMESPACE, BANNER_IMAGES_FORMATS.NAMESPACE,
                        BannerImageFormatNamespace::fromSource, BannerImageFormatNamespace::toSource))
                .map(convertibleProperty(Image.IMAGE_TYPE, BANNER_IMAGES_FORMATS.IMAGE_TYPE,
                        ImageType::fromSource, ImageType::toSource))
                .readProperty(Image.SIZE,
                        fromFields(BANNER_IMAGES_FORMATS.WIDTH, BANNER_IMAGES_FORMATS.HEIGHT)
                                .by(BannerImageMapperProvider::imageSizeFromDb))
                .writeField(BANNER_IMAGES_FORMATS.WIDTH,
                        fromProperty(Image.SIZE).by(BannerImageMapperProvider::dbWidthFromImageSize))
                .writeField(BANNER_IMAGES_FORMATS.HEIGHT,
                        fromProperty(Image.SIZE).by(BannerImageMapperProvider::dbHeightFromImageSize))
                .map(convertibleProperty(Image.FORMATS, BANNER_IMAGES_FORMATS.FORMATS,
                        BannerImageMapperProvider::toImageFormats, JsonUtils::toJsonNullable))
                .map(convertibleProperty(Image.AVATARS_HOST, BANNER_IMAGES_FORMATS.AVATARS_HOST,
                        AvatarHost::fromSource, AvatarHost::toSource))
                .map(convertibleProperty(Image.MDS_META, BANNER_IMAGES_FORMATS.MDS_META,
                        BannerImageConverter::toImageMdsMeta, JsonUtils::toJsonNullable))
                .map(convertibleProperty(Image.MDS_META_USER_OVERRIDE, BANNER_IMAGES_POOL.MDS_META_USER_OVERRIDE,
                        BannerImageConverter::toImageMdsMeta, JsonUtils::toJsonNullable))
                .build();
    }

}
