package ru.yandex.direct.core.entity.image.service;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.core.entity.image.service.validation.ImageDownloadException;
import ru.yandex.direct.core.entity.zora.ZoraService;
import ru.yandex.direct.zorafetcher.ZoraResponse;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.UrlUtils.urlDomainToAscii;

@Component
@ParametersAreNonnullByDefault
public class ImageDownloader {
    private static final Logger logger = LoggerFactory.getLogger(ImageDownloader.class);

    private final ZoraService zoraService;

    @Autowired
    public ImageDownloader(ZoraService zoraService) {
        this.zoraService = zoraService;
    }

    public byte[] download(String url) {
        String urlToFetch = urlDomainToAscii(url);
        Result<ZoraResponse> result = zoraService.fetchByUrl(urlToFetch, true);

        if (!isEmpty(result.getErrors())) {
            result.getErrors().forEach(err -> logger.error("There is an error during image downloading"));
            throw new ImageDownloadException("Received fetch result should not have errors");
        }

        ZoraResponse zoraResponse = checkNotNull(result.getSuccess());
        if (!zoraResponse.isOk()) {
            if (zoraResponse.isNotOkByZora()) {
                logger.warn("url to fetch: '{}', zora response: {\n{}}'",
                        urlToFetch, zoraResponse.prettyZoraResponse());
            } else {
                logger.info("url to fetch: '{}', response with status code {} {}",
                        urlToFetch, zoraResponse.getStatusCode(), zoraResponse.getStatusText());
            }
            throw new ImageDownloadException("Response code should be 200");
        }

        return zoraResponse.getResponse().getResponseBodyAsBytes();
    }
}
