package ru.yandex.direct.core.entity.image.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableBytesRequest;
import ru.yandex.direct.avatars.client.AvatarsClient;
import ru.yandex.direct.avatars.client.exception.AvatarsClientCommonException;
import ru.yandex.direct.avatars.client.model.AvatarInfo;
import ru.yandex.direct.avatars.config.ServerConfig;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsClientPool;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.ImageMetaInformation;
import ru.yandex.direct.core.entity.image.container.UploadedBannerImageInformation;
import ru.yandex.direct.core.entity.image.container.UploadedToMdsImageInformation;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta;
import ru.yandex.direct.core.entity.image.model.ImageUploadContainer;
import ru.yandex.direct.core.entity.image.model.ImageValidationContainer;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.image.repository.BannerImagePoolRepository;
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository;
import ru.yandex.direct.core.entity.image.service.validation.ImageDownloadException;
import ru.yandex.direct.core.entity.image.service.validation.SaveImageFromUrlToMdsValidationService;
import ru.yandex.direct.core.entity.image.service.validation.SaveImageFromUrlValidationService;
import ru.yandex.direct.core.entity.image.service.validation.SaveImageValidationService;
import ru.yandex.direct.core.entity.image.service.validation.type.ImageSaveValidationForTextImageAdSupport;
import ru.yandex.direct.core.entity.image.service.validation.type.ImageSaveValidationSupportFacade;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileMetadata;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileSaveRequest;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.mds.MdsStorageException;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.image.converter.BannerImageConverter.toBannerImageFormat;
import static ru.yandex.direct.core.entity.image.converter.BannerImageConverter.toBannerImageFormatNamespace;
import static ru.yandex.direct.core.entity.image.converter.BannerImageConverter.toBannerImageFromPoolInformation;
import static ru.yandex.direct.core.entity.image.converter.BannerImageConverter.toImageMdsMeta;
import static ru.yandex.direct.core.entity.image.converter.BannerImageConverter.toImageType;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_SMALL_IMAGE_MIN_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MIN_LOGO_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.ORIG;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.collectImageMetaInformation;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.extractImageNameFromUrl;
import static ru.yandex.direct.core.entity.image.service.validation.ImageDefects.invalidImageFile;
import static ru.yandex.direct.core.entity.image.service.validation.ImageDefects.invalidImageUrl;
import static ru.yandex.direct.feature.FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HashAsBase64YaStringWithoutPadding;
import static ru.yandex.direct.validation.result.PathHelper.index;

@Service
@ParametersAreNonnullByDefault
public class ImageService {
    private static final Logger logger = LoggerFactory.getLogger(ImageService.class);
    private static final Set IMAGE_TYPES_TO_GET_FROM_TEXT_AVATARS_CLIENTS_POOL = Set.of(
            BannerImageType.BANNER_TEXT, BannerImageType.ASSET_LOGO, BannerImageType.ASSET_LOGO_EXTENDED,
            BannerImageType.ASSET_MULTICARD, BannerImageType.OFFER_IMAGE);
    public static final int LOGO_SIZE = MIN_LOGO_SIZE;

    private final ImageSaveValidationSupportFacade imageSaveValidationSupportFacade;

    private final AvatarsClientPool textBannerImagesAvatarsClientPool;
    private final AvatarsClientPool imageBannerImagesAvatarsClientPool;
    private final ImageDownloader imageDownloader;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;

    private final SaveImageFromUrlToMdsValidationService saveImageFromUrlToMdsValidationService;
    private final SaveImageFromUrlValidationService saveImageFromUrlValidationService;
    private final SaveImageValidationService saveImageValidationService;

    private final MdsFileService mdsFileService;

    private final BannerImagePoolRepository bannerImagePoolRepository;
    private final BannerImageFormatRepository bannerImageFormatRepository;

    private final BannerImageRepository bannerImageRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final ImageDataRepository imageDataRepository;

    private final ParallelFetcherFactory parallelFetcherFactory;

    @Autowired
    public ImageService(ImageSaveValidationSupportFacade imageSaveValidationSupportFacade,
                        AvatarsClientPool textBannerImagesAvatarsClientPool,
                        AvatarsClientPool imageBannerImagesAvatarsClientPool,
                        ImageDownloader imageDownloader,
                        AsyncHttpClient asyncHttpClient,
                        ShardHelper shardHelper,
                        FeatureService featureService, SaveImageFromUrlToMdsValidationService saveImageFromUrlToMdsValidationService,
                        SaveImageFromUrlValidationService saveImageFromUrlValidationService,
                        SaveImageValidationService saveImageValidationService,
                        MdsFileService mdsFileService,
                        BannerImagePoolRepository bannerImagePoolRepository,
                        BannerImageFormatRepository bannerImageFormatRepository,
                        BannerImageRepository bannerImageRepository,
                        BannerCommonRepository bannerCommonRepository,
                        ImageDataRepository imageDataRepository) {
        this.imageSaveValidationSupportFacade = imageSaveValidationSupportFacade;

        this.textBannerImagesAvatarsClientPool = textBannerImagesAvatarsClientPool;
        this.imageBannerImagesAvatarsClientPool = imageBannerImagesAvatarsClientPool;
        this.imageDownloader = imageDownloader;
        this.shardHelper = shardHelper;
        this.featureService = featureService;

        this.saveImageFromUrlToMdsValidationService = saveImageFromUrlToMdsValidationService;
        this.saveImageFromUrlValidationService = saveImageFromUrlValidationService;
        this.saveImageValidationService = saveImageValidationService;

        this.mdsFileService = mdsFileService;

        this.bannerImagePoolRepository = bannerImagePoolRepository;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.bannerImageRepository = bannerImageRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.imageDataRepository = imageDataRepository;

        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient,
                new FetcherSettings().withRequestRetries(3));
    }

    /**
     * Выгружает по ссылке файл.
     * Если тип файла: jpg, jpeg, png, gif -- сохраняем в MDS(не в аватарницу!).
     * Ошибки валидации прикрепляются к ссылке.
     *
     * @param clientId id клиента
     * @param url      ссылка по которой скачиваем изображение
     */
    public Result<UploadedToMdsImageInformation> saveImageFromUrlToMds(ClientId clientId, String url) {
        ValidationResult<String, Defect> vr = saveImageFromUrlToMdsValidationService.preValidate(url);

        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        byte[] fileData;
        try {
            fileData = imageDownloader.download(url);
        } catch (ImageDownloadException e) {
            return Result.broken(vr.addError(invalidImageUrl()));
        }

        saveImageFromUrlToMdsValidationService.validate(vr, fileData);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        String imageHash = getMd5HashAsBase64YaStringWithoutPadding(fileData);
        String imageNameFromUrl = extractImageNameFromUrl(url);

        Map<String, MdsFileMetadata> metaData =
                mdsFileService.getMetaData(clientId, Collections.singletonList(imageHash));
        boolean clientAlreadyUploadImage = metaData.containsKey(imageHash);
        if (clientAlreadyUploadImage) {
            return collectSaveImageFromUrlResult(vr, imageHash, imageNameFromUrl);
        }

        MdsFileSaveRequest mdsFileSaveRequest =
                new MdsFileSaveRequest(MdsStorageType.BANNER_IMAGES_UPLOADS, fileData, imageHash);

        try {
            List<MdsFileSaveRequest> saveResult =
                    mdsFileService.saveMdsFiles(Collections.singletonList(mdsFileSaveRequest), clientId.asLong());
            checkState(saveResult.size() == 1, "Result size should be 1");
        } catch (MdsStorageException e) {
            logger.info("Can not upload image to mds: ", e);
            return Result.broken(vr.addError(invalidImageFile()));
        }

        return collectSaveImageFromUrlResult(vr, imageHash, imageNameFromUrl);
    }

    private Result<UploadedToMdsImageInformation> collectSaveImageFromUrlResult(ValidationResult<String, Defect> vr,
                                                                                String imageHash,
                                                                                String imageNameFromUrl) {
        UploadedToMdsImageInformation result = new UploadedToMdsImageInformation()
                .withName(imageNameFromUrl)
                .withImageHash(imageHash);
        return Result.successful(result, vr);
    }

    /**
     * Загружает изображения по ссылке и сохраняет в аватарницу.
     * Ошибки валидации прикрепляются к ссылке.
     *
     * @param clientId        id клиента
     * @param url             ссылка по которой скачиваем изображение
     * @param bannerImageType тип проверки картинки
     */
    public Result<UploadedBannerImageInformation> saveImageFromUrl(
            ClientId clientId, String url,
            BannerImageType bannerImageType,
            BannerImageSource source,
            @Nullable ResourceInfo resourceInfo) {
        ValidationResult<String, Defect> vr = saveImageFromUrlValidationService.validate(url);

        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        byte[] fileData;
        ImageMetaInformation imageMetaInformation;
        try {
            fileData = imageDownloader.download(url);
        } catch (ImageDownloadException e) {
            return Result.broken(vr.addError(invalidImageUrl()));
        }

        saveImageValidationService.validateMimeType(vr, fileData);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        try {
            imageMetaInformation = collectImageMetaInformation(fileData);
        } catch (IOException e) {
            return Result.broken(vr.addError(invalidImageFile()));
        }

        String imageNameFromUrl = extractImageNameFromUrl(url);

        ValidationResult<String, Defect> vrImageInformation =
                imageSaveValidationSupportFacade.getImageSaveSupport(bannerImageType)
                        .validateImageMetaInformation(url, imageMetaInformation);

        vr.merge(vrImageInformation);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        int shard = shardHelper.getShardByClientId(clientId);
        return uploadImage(shard, clientId, bannerImageType, imageMetaInformation.getSize(), vr, imageNameFromUrl,
                fileData, getAvatarsClient(bannerImageType), source, resourceInfo);
    }

    /**
     * Сохраняет изображения для различных типов баннеров. Изображениям присваиваем index,
     * и далее все ошибки валидации привязываем к индексу каждого изображения.
     *
     * @param clientId        id клиента
     * @param uploadContainersById   мапа файлов по идентификаторам
     * @param bannerImageType тип проверки картинки
     */
    public MassResult<UploadedBannerImageInformation> saveImages(ClientId clientId,
                                                                 Map<Integer, ImageUploadContainer> uploadContainersById,
                                                                 BannerImageType bannerImageType) {
        List<Integer> fileIds = new ArrayList<>(uploadContainersById.keySet());
        List<MultipartFile> files = mapList(uploadContainersById.values(), ImageUploadContainer::getFile);
        ValidationResult<List<Integer>, Defect> vr = new ValidationResult<>(fileIds);

        List<byte[]> imageDataList = new ArrayList<>(files.size());

        Map<Integer, byte[]> fileDataById = fetchFileData(fileIds, files, vr);

        saveImageValidationService.validateMimeType(vr, fileDataById);

        Map<Integer, Integer> imageIdToImageIndex = IntStreamEx.range(fileIds.size())
                .boxed()
                .mapToEntry(fileIds::get)
                .invert()
                .toMap();

        Map<Integer, Integer> imageIdToIndexOfFetchedImage = new HashMap<>();
        List<ImageMetaInformation> imageMetaInformationList = collectImageMetaInformationList(vr, imageDataList,
                imageIdToIndexOfFetchedImage, fileDataById, imageIdToImageIndex);

        imageSaveValidationSupportFacade
                .validate(vr, imageIdToIndexOfFetchedImage, bannerImageType, imageMetaInformationList);

        List<ImageSize> imageSizes = mapList(imageMetaInformationList, ImageMetaInformation::getSize);
        List<String> imageNames = mapList(files, MultipartFile::getOriginalFilename);

        return uploadImages(clientId, bannerImageType, imageDataList, imageSizes, imageNames, vr,
                imageIdToIndexOfFetchedImage, imageIdToImageIndex, uploadContainersById);
    }

    public Map<String, BannerImageFromPool> getBannerImageFromPoolsByHashes(ClientId clientId,
                                                                            Collection<String> imageHashes) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return bannerImagePoolRepository.getBannerImageFromPoolsByHashes(shard, clientId, imageHashes);
    }


    public List<BannerImageFormat> getBannerImageFormats(ClientId clientId, Collection<String> imageHashes) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<String, BannerImageFormat> bannerImageFormats
                = bannerImageFormatRepository.getBannerImageFormats(shard, clientId, imageHashes);
        return List.copyOf(bannerImageFormats.values());
    }

    /**
     * Метод собирает мета информацию о файлах для сохранения.
     * Если получилось получить такую информацию, то добавляем:
     * в <code>imageIdToIndexOfFetchedImage</code> пару - id файла и его порядковый номер в результатирующем списке
     * в <code>imageDataList</code> исходный массив байт файла
     *
     * @param vr
     * @param imageDataList - пустой список
     * @param imageIdToIndexOfFetchedImage - пустая мапа
     * @param imageDataById - мапа с данными файла по ключу
     * @param imageIdToImageIndex - отношение между ид файла и его позицией в коллекции
     * @return
     */
    private List<ImageMetaInformation> collectImageMetaInformationList(ValidationResult<List<Integer>, Defect> vr,
                                                                       List<byte[]> imageDataList,
                                                                       Map<Integer, Integer> imageIdToIndexOfFetchedImage,
                                                                       Map<Integer, byte[]> imageDataById,
                                                                       Map<Integer, Integer> imageIdToImageIndex) {
        List<ImageMetaInformation> imageMetaInformationList = new ArrayList<>(imageIdToImageIndex.size());
        int indexOfFetchedImage = 0;
        List<Integer> supportedImageIds = ValidationResult.getValidItems(vr);
        for (Integer imageId : supportedImageIds) {
            try {
                byte[] imageData = imageDataById.get(imageId);
                ImageMetaInformation imageMetaInformation = collectImageMetaInformation(imageData);

                imageDataList.add(imageData);
                imageMetaInformationList.add(imageMetaInformation);
                imageIdToIndexOfFetchedImage.put(imageId, indexOfFetchedImage);
                indexOfFetchedImage++;
            } catch (IOException e) {
                logger.error("Image with id: " + imageId + " failed on collecting image meta information", e);
                vr.getOrCreateSubValidationResult(index(imageIdToImageIndex.get(imageId)), imageId)
                        .addError(invalidImageFile());
            }
        }
        return imageMetaInformationList;
    }

    private Map<Integer, byte[]> fetchFileData(List<Integer> fileIds, List<MultipartFile> files,
                                               ValidationResult<List<Integer>, Defect> vr) {
        Map<Integer, byte[]> dataById = new HashMap<>();
        for (int i = 0; i < files.size(); i++) {
            Integer fileId = fileIds.get(i);
            MultipartFile file = files.get(i);
            try {
                byte[] bytes = file.getBytes();
                dataById.put(fileId, bytes);
            } catch (IOException e) {
                logger.error("File with id: " + fileId +
                        "and name: " + file.getOriginalFilename() + " failed on fetching", e);
                vr.getOrCreateSubValidationResult(index(i), fileId)
                        .addError(invalidImageFile());
            }
        }
        return dataById;
    }

    private MassResult<UploadedBannerImageInformation> uploadImages(ClientId clientId, BannerImageType bannerImageType,
                                                                    List<byte[]> imageDataList,
                                                                    List<ImageSize> imageSizes, List<String> imageNames,
                                                                    ValidationResult<List<Integer>, Defect> vr,
                                                                    Map<Integer, Integer> indexOfFetchedImageByImageId,
                                                                    Map<Integer, Integer> imageIndexByImageId,
                                                                    Map<Integer, ImageUploadContainer> uploadContainersById) {
        int shard = shardHelper.getShardByClientId(clientId);
        AvatarsClient avatarsClient;
        avatarsClient = getAvatarsClient(bannerImageType);

        List<Result<UploadedBannerImageInformation>> uploadedBannerImageInformationList =
                EntryStream.of(indexOfFetchedImageByImageId)
                        .mapKeyValue((id, indexOfFetched) -> {
                            Integer index = imageIndexByImageId.get(id);
                            ValidationResult<Integer, Defect> vrElement =
                                    vr.getOrCreateSubValidationResult(index(index), id);
                            if (vrElement.hasAnyErrors()) {
                                return Result.<UploadedBannerImageInformation>broken(vr);
                            }

                            ImageSize imageSize = imageSizes.get(indexOfFetched);
                            String originalFilename = imageNames.get(indexOfFetched);
                            byte[] imageData = imageDataList.get(indexOfFetched);

                            return uploadImage(shard, clientId, bannerImageType, imageSize, vrElement, originalFilename,
                                    imageData, avatarsClient, BannerImageSource.DIRECT,
                                    uploadContainersById.get(id).getResource());
                        })
                        .toList();
        boolean hasSuccessfulResults = uploadedBannerImageInformationList.stream()
                .map(Result::getState)
                .anyMatch(ResultState.SUCCESSFUL::equals);
        return new MassResult<>(uploadedBannerImageInformationList, vr,
                hasSuccessfulResults ? ResultState.SUCCESSFUL : ResultState.BROKEN);
    }

    private AvatarsClient getAvatarsClient(BannerImageType bannerImageType) {
        if (IMAGE_TYPES_TO_GET_FROM_TEXT_AVATARS_CLIENTS_POOL.contains(bannerImageType)) {
            return textBannerImagesAvatarsClientPool.getDefaultClient();
        }
        return imageBannerImagesAvatarsClientPool.getDefaultClient();
    }


    private <T> Result<UploadedBannerImageInformation> uploadImage(
            int shard, ClientId clientId,
            BannerImageType bannerImageType,
            ImageSize imageSize,
            ValidationResult<T, Defect> vr,
            String originalFilename,
            byte[] imageData,
            AvatarsClient avatarsClient,
            BannerImageSource source,
            @Nullable ResourceInfo resourceInfo) {
        String hash = getMd5HashAsBase64YaStringWithoutPadding(imageData);

        if (bannerImageType == BannerImageType.ASSET_LOGO && imageSize.getWidth() >= BANNER_SMALL_IMAGE_MIN_SIZE) {
            imageData = resizeLogo(imageData, avatarsClient, hash);
            hash = getMd5HashAsBase64YaStringWithoutPadding(imageData);
            imageSize = new ImageSize().withWidth(LOGO_SIZE).withHeight(LOGO_SIZE);
        }

        Map<String, BannerImageFormat> bannerImageFormatByHash =
                bannerImageFormatRepository.getBannerImageFormats(shard, Collections.singletonList(hash));
        boolean isAlreadyUploadedToClientShard = bannerImageFormatByHash.containsKey(hash);

        BannerImageFromPool bannerImageFromPool =
                toBannerImageFromPoolInformation(clientId.asLong(), originalFilename, hash, source);
        bannerImagePoolRepository
                .addOrUpdateImagesToPool(shard, clientId, Collections.singletonList(bannerImageFromPool));

        if (isAlreadyUploadedToClientShard) {
            BannerImageFormat bannerImageFormat = bannerImageFormatByHash.get(hash);
            var vrBeforeSave = imageSaveValidationSupportFacade.getImageSaveSupport(bannerImageType)
                    .validateBeforeSave(vr.getValue(), new ImageValidationContainer(bannerImageFormat, resourceInfo));
            if (vrBeforeSave.hasAnyErrors()) {
                return Result.broken(vrBeforeSave);
            }
            if (!imageSize.getWidth().equals(bannerImageFormat.getSize().getWidth()) ||
                    !imageSize.getHeight().equals(bannerImageFormat.getSize().getHeight())) {

                bannerImageFormat.setSize(imageSize);
                bannerImageFormatRepository.updateBannerImageFormatSize(shard, bannerImageFormat);
            }
            return collectUploadedTextBannerImageInformation(vr, bannerImageType, originalFilename, hash,
                    bannerImageFormat.getMdsGroupId(), bannerImageFormat.getNamespace(), bannerImageFormat.getSize(),
                    bannerImageFormat.getMdsMeta());
        }

        AvatarInfo avatarInfo;
        //invalidImageFile -> change to unknownError
        try {
            avatarInfo = avatarsClient.upload(hash, imageData);
            if (!avatarInfo.getSizes().containsKey(ORIG)) {
                logger.info("No orig format for imageHash: {}", hash);
                return Result.broken(vr.addError(invalidImageFile()));
            }

            ru.yandex.direct.avatars.client.model.answer.ImageSize originalImageSizeFromAvatar =
                    avatarInfo.getSizes().get(ORIG);
            boolean allowProportionallyLargerImages =
                    featureService.isEnabledForClientId(clientId, ALLOW_PROPORTIONALLY_LARGER_IMAGES);
            boolean notEqualSize = originalImageSizeFromAvatar.getWidth() != imageSize.getWidth()
                    || originalImageSizeFromAvatar.getHeight() != imageSize.getHeight();
            ImageSize toCheckImageSize = new ImageSize()
                    .withWidth(originalImageSizeFromAvatar.getWidth())
                    .withHeight(originalImageSizeFromAvatar.getHeight());

            if (notEqualSize && (!allowProportionallyLargerImages
                    || !isProportionallyLarger(toCheckImageSize, bannerImageType))) {
                logger.info("mds returned invalid size for imageHash: {}", hash);
                return Result.broken(vr.addError(invalidImageFile()));
            }

        } catch (AvatarsClientCommonException e) {
            logger.info("Can not save image with hash:" + hash + "to avatarMds", e);
            return Result.broken(vr.addError(invalidImageFile()));
        }

        BannerImageFormat bannerImageFormat = toBannerImageFormat(imageSize, bannerImageType,
                avatarsClient.getConf().getReadServerConfig().getServerHost(), hash, avatarInfo);
        if (bannerImageFormat.getFormats().isEmpty()) {
            logger.warn("Can not save image (hash {}) without formats", hash);
            return Result.broken(vr.addError(invalidImageFile()));
        }

        var vrBeforeSave = imageSaveValidationSupportFacade.getImageSaveSupport(bannerImageType)
                .validateBeforeSave(vr.getValue(), new ImageValidationContainer(bannerImageFormat, resourceInfo));
        if (vrBeforeSave.hasAnyErrors()) {
            return Result.broken(vrBeforeSave);
        }

        bannerImageFormatRepository.addBannerImageFormat(shard, Collections.singletonList(bannerImageFormat));
        //noinspection ConstantConditions
        return collectUploadedTextBannerImageInformation(vr, bannerImageType, originalFilename, hash,
                avatarInfo.getGroupId(), toBannerImageFormatNamespace(avatarInfo.getNamespace()), imageSize,
                bannerImageFormat.getMdsMeta());
    }

    /* Логотип может пересекаться с типами wide и small (в БД - колонка banner_images_formats.image_type).
       В результате такие картинки нельзя будет использовать как картинку и как логотип.
       Чтобы этого не было, резайзим логотип до 80 пикселей.
       В перспективе, возможно, имеет смысл отказаться от этой колонки в БД или заменить ее на set
    */
    private byte[] resizeLogo(byte[] imageData, AvatarsClient avatarsClient, String hash) {
        AvatarInfo avatarInfo = avatarsClient.upload(hash, imageData);
        ImageMdsMeta imageMdsMeta = ifNotNull(avatarInfo.getMeta(), j -> JsonUtils.fromJson(j, ImageMdsMeta.class));
        String path = imageMdsMeta.getSizes().get("x" + LOGO_SIZE).getPath();

        ServerConfig readServerConfig = avatarsClient.getConf().getReadServerConfig();
        String url = readServerConfig.getServerSchema() + "://" + readServerConfig.getServerHost() + path;

        try (ParallelFetcher<ParsableBytesRequest.Byteswrapper> fetcher = parallelFetcherFactory.getParallelFetcher()) {
            ParsableBytesRequest request = new ParsableBytesRequest(0, new RequestBuilder().setUrl(url).build());
            return fetcher.execute(request).getSuccess().getContent();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private <T> Result<UploadedBannerImageInformation> collectUploadedTextBannerImageInformation(
            ValidationResult<T, Defect> vr, BannerImageType bannerImageType, String name, String imageHash,
            int mdsGroupId, BannerImageFormatNamespace namespace, ImageSize imageSize, String meta) {
        return Result.successful(new UploadedBannerImageInformation()
                .withImageHash(imageHash)
                .withMdsGroupId(mdsGroupId)
                .withNamespace(namespace)
                .withName(name)
                .withWidth(imageSize.getWidth())
                .withHeight(imageSize.getHeight())
                .withMdsMeta(toImageMdsMeta(meta))
                .withImageType(toImageType(imageSize, bannerImageType)), vr);
    }

    public void saveOverriddenSmartCenters(int shard, ClientId clientId, String imageHash,
                                           ImageMdsMeta overriddenSmartCenters) {
        imageDataRepository
                .overrideImageSmartCenters(shard, clientId, imageHash, JsonUtils.toJson(overriddenSmartCenters));

        resetStatusBsSynced(shard, imageHash, clientId);
    }

    private void resetStatusBsSynced(int shard, String imageHash, ClientId clientId) {
        Collection<Long> bids = bannerImageRepository
                .getBidsByImageHash(shard, singletonList(imageHash), clientId.asLong());
        bannerCommonRepository.resetStatusBsSyncedByIds(shard, bids);
    }

    private boolean isProportionallyLarger(ImageSize imageSize, BannerImageType bannerImageType) {
        if (bannerImageType == BannerImageType.BANNER_IMAGE_AD) {
            var support = (ImageSaveValidationForTextImageAdSupport) imageSaveValidationSupportFacade
                    .getImageSaveSupport(bannerImageType);
            return support.isProportionallyLarger(imageSize);
        }
        return false;
    }
}
