package ru.yandex.direct.core.entity.image.service;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.regex.Matcher;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.imageio.IIOException;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import org.apache.tika.Tika;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.core.entity.image.container.ImageMetaInformation;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.ACCURACY;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.AVATAR_NAMESPACE_PREFIX_REGEX;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_IMAGE_RATIO;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_IMAGE_WIDE_MIN_WIDTH_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_IMAGE_WIDE_RATIO;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_REGULAR_IMAGE_MIN_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_SMALL_IMAGE_MIN_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.GIF_MIME_TYPE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.LAST_URL_SEGMENT_PATTERN;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_LOGO_EXTENDED_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MIN_LOGO_EXTENDED_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MIN_LOGO_SIZE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.ORIG;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SVG_MIME_TYPE;

@ParametersAreNonnullByDefault
public class ImageUtils {
    private static final Tika MIME_TYPE_DETECTOR = new Tika();

    public static String extractImageNameFromUrl(String url) {
        Matcher matcher = LAST_URL_SEGMENT_PATTERN.matcher(url);
        checkState(matcher.find(), "url should have special format");
        return matcher.group();
    }

    public static ImageMdsMeta copyImageMdsMeta(ImageMdsMeta imageMdsMeta) {
        String json = JsonUtils.toJson(imageMdsMeta);
        return JsonUtils.fromJson(json, ImageMdsMeta.class);
    }

    /**
     * Собирает необходимую информацию об изображении
     */
    public static ImageMetaInformation collectImageMetaInformation(byte[] imageData) throws IOException {
        String mimeType = getMimeType(imageData);
        ImageSize imageSize = getImageSize(imageData, mimeType);
        int framesNumber = getFramesNumber(imageData, mimeType);
        return new ImageMetaInformation()
                .withFormat(ImageFileFormat.fromTypedValue(mimeType))
                .withSize(imageSize)
                .withImageFileSize(imageData.length)
                .withFramesNumber(framesNumber);
    }

    /**
     * Получает размер изображения
     */
    private static ImageSize getImageSize(byte[] imageData, String mimeType) throws IOException {
        if (mimeType.equals(SVG_MIME_TYPE)) {
            return SvgImageUtils.getSvgImageSize(imageData)
                    .orElseThrow(() -> new IOException("Could not read SVG size"));
        }
        try (InputStream inputStream = new ByteArrayInputStream(imageData)) {
            BufferedImage image = ImageIO.read(inputStream);

            return new ImageSize()
                    .withWidth(image.getWidth())
                    .withHeight(image.getHeight());
        } catch (ArrayIndexOutOfBoundsException | IIOException ex) {
            // некоторые валидные gif-файлы не читаются из-за бага JDK JDK-8055047,
            // для этого делаем workaround - не читая файл читаем только матаданные
            if (Arrays.stream(ex.getStackTrace()).anyMatch(f -> f.getClassName().contains("GIFImageReader"))) {
                return getGifImageSize(imageData);
            } else {
                throw ex;
            }
        }
    }

    private static ImageSize getGifImageSize(byte[] imageData) throws IOException {
        try (InputStream inputStream = new ByteArrayInputStream(imageData)) {
            ImageReader reader = ImageIO.getImageReadersByMIMEType(GIF_MIME_TYPE).next();
            ImageInputStream imageInputStream = ImageIO.createImageInputStream(inputStream);
            reader.setInput(imageInputStream);

            return new ImageSize()
                    .withWidth(reader.getWidth(0))
                    .withHeight(reader.getHeight(0));
        }
    }

    /**
     * Получает тип изображения
     */
    public static String getMimeType(byte[] imageData) {
        String detectedMimeType = MIME_TYPE_DETECTOR.detect(imageData);
        if (detectedMimeType.equals("text/plain") && SvgImageUtils.isSvgWithoutLeadingXmlTag(imageData)) {
            // Некоторые SVG файлы не начинаются с тэга `xml`, т.е. не являются валидным XML,
            // что не мешает браузерами и пользователям с ними работать
            // поэтому мы тоже должны их поддерживать
            return SVG_MIME_TYPE;
        }
        return detectedMimeType;
    }

    /**
     * TODO Временное решение. Улучшим тут: DIRECT-92085
     */
    public static String toDirectImagePath(String path) {
        return path.replaceFirst(AVATAR_NAMESPACE_PREFIX_REGEX, "/");
    }


    /**
     * Получает числа кадров в изображении.
     * Для gif может быть больше 1.
     */
    private static int getFramesNumber(byte[] imageData, String mimeType) throws IOException {
        try (InputStream inputStream = new ByteArrayInputStream(imageData)) {
            if (GIF_MIME_TYPE.equals(mimeType)) {
                ImageReader imageReader = ImageIO.getImageReadersByMIMEType(GIF_MIME_TYPE).next();
                ImageInputStream imageInputStream = ImageIO.createImageInputStream(inputStream);
                imageReader.setInput(imageInputStream);
                return imageReader.getNumImages(true);
            }
            return 1;
        }
    }

    /**
     * По переданным размерам вычисляет тип изображения.
     */
    public static ImageType calculateImageTypeOfTextBannerImage(ImageSize imageSize) {
        int width = imageSize.getWidth();
        int height = imageSize.getHeight();
        checkArgument(isSizeSupported(width, height), "size is not supported");

        if (isWideImage(width, height)) {
            return ImageType.WIDE;
        }

        if (isSmallImage(width, height)) {
            return ImageType.SMALL;
        }

        return ImageType.REGULAR;
    }

    public static ImageType calculateImageTypeOfLogoExtendedImage(ImageSize imageSize) {
        int width = imageSize.getWidth();
        int height = imageSize.getHeight();
        checkArgument(isSizeAvailableForLogoExtended(width, height), "size is not supported");

        if (isSizeSupported(width, height)) {
            return calculateImageTypeOfTextBannerImage(imageSize);
        }

        return ImageType.LOGO;
    }

    public static boolean isSizeSupported(int width, int height) {
        return isWideImage(width, height) || isSmallImage(width, height) || isRegularImage(width, height);
    }

    public static boolean isSizeAvailableForNewImages(int width, int height) {
        return isWideImage(width, height) || isRegularImage(width, height);
    }

    public static boolean isSizeAvailableForLogo(int width, int height) {
        return width == height && width >= MIN_LOGO_SIZE;
    }

    public static boolean isSizeAvailableForLogoExtended(int width, int height) {
        return width >= MIN_LOGO_EXTENDED_SIZE && width <= MAX_LOGO_EXTENDED_SIZE
                && height >= MIN_LOGO_EXTENDED_SIZE && height <= MAX_LOGO_EXTENDED_SIZE;
    }

    /**
     * Изображение считается широким, если ширина превышает 1080,
     * и больше высоты в 16/9 раз с точность до половины пикселя.
     */
    private static boolean isWideImage(int width, int height) {
        return width >= BANNER_IMAGE_WIDE_MIN_WIDTH_SIZE &&
                Math.abs(width / BANNER_IMAGE_WIDE_RATIO - height) < ACCURACY;
    }

    /**
     * Изображение считается малым, если меньшая сторона изображения находится в промежутке [150; 450),
     * а соотношение сторон является стандартным
     */
    private static boolean isSmallImage(int width, int height) {
        long smallerSide = Math.min(width, height);
        return smallerSide < BANNER_REGULAR_IMAGE_MIN_SIZE && smallerSide >= BANNER_SMALL_IMAGE_MIN_SIZE &&
                isRegularRatio(width, height);
    }

    /**
     * Изображение считается стандартным, если обе стороны больше либо равны чем 450,
     * а соотношение сторон является стандартным
     */
    private static boolean isRegularImage(int width, int height) {
        return width >= BANNER_REGULAR_IMAGE_MIN_SIZE && height >= BANNER_REGULAR_IMAGE_MIN_SIZE &&
                isRegularRatio(width, height);
    }

    /**
     * Соотношение сторон является стандартным, если большая сторона превышает меньшую сторону менее чем в 4/3 раз.
     */
    private static boolean isRegularRatio(int width, int height) {
        return (double) width / height < BANNER_IMAGE_RATIO ||
                (double) height / width < BANNER_IMAGE_RATIO;
    }

    /**
     * Генерирует ссылку для оригинальной картинки, загруженной в аватарницу
     */
    public static String generateOrigImageUrl(BannerImageFormat bannerImageFormat) {
        return generateImageUrl(bannerImageFormat.getAvatarsHost(), bannerImageFormat.getNamespace(),
                bannerImageFormat.getMdsGroupId(), bannerImageFormat.getImageHash(), ORIG);
    }

    /**
     * Генерирует https ссылку для оригинальной картинки, загруженной в аватарницу
     */
    public static String generateSecureOrigImageUrl(BannerImageFormat bannerImageFormat) {
        return generateImageSecureUrl(bannerImageFormat.getAvatarsHost(), bannerImageFormat.getNamespace(),
                bannerImageFormat.getMdsGroupId(), bannerImageFormat.getImageHash(), ORIG);
    }

    /**
     * Генерирует ссылку для указанного формата картинки, загруженной в аватарницу
     */
    public static String generateImageUrl(BannerImageFormat bannerImageFormat, String formatId) {
        return generateImageUrl(bannerImageFormat.getAvatarsHost(), bannerImageFormat.getNamespace(),
                bannerImageFormat.getMdsGroupId(), bannerImageFormat.getImageHash(), formatId);
    }

    /**
     * Генерирует {@code https} ссылку для указанного формата картинки, загруженной в аватарницу
     */
    public static String generateImageUrl(Image image, String formatId) {
        return generateImageSecureUrl(image.getAvatarsHost(), image.getNamespace(), image.getMdsGroupId(),
                image.getImageHash(), formatId);
    }

    public static String generateImageSecureUrl(BannerImageFormat bannerImageFormat, String formatId) {
        return generateImageSecureUrl(bannerImageFormat.getAvatarsHost(), bannerImageFormat.getNamespace(),
                bannerImageFormat.getMdsGroupId(), bannerImageFormat.getImageHash(), formatId);
    }

    private static String generateImageUrl(AvatarHost avatarHost, BannerImageFormatNamespace avatarNamespace,
                                           Integer mdsGroupId, String imageHash, String formatId) {
        return generateImageUrl("http", avatarHost, avatarNamespace, mdsGroupId, imageHash, formatId);
    }

    private static String generateImageSecureUrl(AvatarHost avatarHost, BannerImageFormatNamespace avatarNamespace,
                                                 Integer mdsGroupId, String imageHash, String formatId) {
        return generateImageUrl("https", avatarHost, avatarNamespace, mdsGroupId, imageHash, formatId);
    }

    private static String generateImageUrl(String protocol, AvatarHost avatarHost,
                                           BannerImageFormatNamespace avatarNamespace,
                                           Integer mdsGroupId, String imageHash, String formatId) {
        String host = checkNotNull(AvatarHost.toSource(avatarHost)).getLiteral();
        String namespace = checkNotNull(BannerImageFormatNamespace.toSource(avatarNamespace)).getLiteral();
        return String.format("%s://%s/get-%s/%s/%s/%s", protocol, host, namespace, mdsGroupId, imageHash, formatId);
    }

}
