package ru.yandex.direct.core.entity.image.service;

import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.regex.Pattern;

import ru.yandex.direct.core.entity.banner.model.ImageSize;

class SvgImageUtils {
    private static final Pattern SVG_WITHOUT_XML_TAG_RE =
            Pattern.compile("^\\s*<svg\\b");

    // должно быть какое-то разумное ограничение на размер аттрибутов
    // считаем, что все атрибуты умещаются в 1024, если нет -- не будем это обрабатывать
    private static final Pattern SVG_ATTRIBUTES_RE =
            Pattern.compile("<svg([^>]{0,1024})>", Pattern.MULTILINE);
    private static final Pattern SVG_WIDTH_RE =
            Pattern.compile("width\\s*=\\s*[\"']\\s*(\\d+)(em|ex|px|pt|pc|cm|mm)?\\s*[\"']", Pattern.MULTILINE);
    private static final Pattern SVG_HEIGHT_RE =
            Pattern.compile("height\\s*=\\s*[\"']\\s*(\\d+)(em|ex|px|pt|pc|cm|mm)?\\s*[\"']", Pattern.MULTILINE);
    private static final Pattern SVG_VIEW_BOX =
            Pattern.compile("viewBox\\s*=\\s*[\"']" +
                    "\\s*\\d+(\\.\\d*)?\\s+\\d+(\\.\\d*)?\\s+(\\d+(\\.\\d*)?)\\s+(\\d+(\\.\\d*)?)\\s*" +
                    "[\"']", Pattern.MULTILINE);

    static Optional<ImageSize> getSvgImageSize(byte[] imageData) {
        // в svg, в корневом тэге могут быть атрибуты width/height и viewBox
        // width и height мы хотим увидеть или в `px` или без указание единиц изменения, что тоже означает тоже `px`
        var content = new String(imageData, StandardCharsets.UTF_8);
        var attrMatcher = SVG_ATTRIBUTES_RE.matcher(content);
        if (!attrMatcher.find()) {
            return Optional.empty();
        }

        String attributes = attrMatcher.group(1);
        Integer width = null;
        Integer height = null;

        // сначала пробуем найти явные атрибуты геометрических размеров
        var widthMatcher = SVG_WIDTH_RE.matcher(attributes);
        if (widthMatcher.find()) {
            if (widthMatcher.group(2) != null && !widthMatcher.group(2).equals("px")) {
                // обрабатываем только пиксельные размеры
                return Optional.empty();
            }
            width = Integer.valueOf(widthMatcher.group(1));
        }
        var heightMatcher = SVG_HEIGHT_RE.matcher(attributes);
        if (heightMatcher.find()) {
            if (widthMatcher.group(2) != null && !widthMatcher.group(2).equals("px")) {
                // обрабатываем только пиксельные размеры
                return Optional.empty();
            }
            height = Integer.valueOf(heightMatcher.group(1));
        }

        // попробуем вытащить размеры из viewBox -- это согласуется с видимым поведением аватарницы
        if (width == null || height == null) {
            var viewBoxMatcher = SVG_VIEW_BOX.matcher(attributes);
            if (viewBoxMatcher.find()) {
                width = Math.round(Float.parseFloat(viewBoxMatcher.group(3)));
                height = Math.round(Float.parseFloat(viewBoxMatcher.group(5)));
            }
        }
        // если никакие способы не сработали -- сдаёмся
        // специализированные библиотеки всё ещё могут определить размер в таком случае,
        // но это требует интерпретации всего содержимого svg
        if (width == null || height == null) {
            return Optional.empty();
        }
        return Optional.of(new ImageSize().withWidth(width).withHeight(height));
    }

    static boolean isSvgWithoutLeadingXmlTag(byte[] imageData) {
        var content = new String(imageData, StandardCharsets.UTF_8);
        return SVG_WITHOUT_XML_TAG_RE.matcher(content).find();
    }
}
