package ru.yandex.direct.core.entity.image.service.validation;

import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_LINEAR_SIZE_IN_PIXEL;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.isSupportedImageMimeType;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@ParametersAreNonnullByDefault
public class ImageConstraints {

    //типы баннеров поддерживающие изображения
    public static final Set<BannersBannerType> BANNER_TYPES_SUPPORTS_IMAGE =
            Set.of(BannersBannerType.text,
                    BannersBannerType.image_ad,
                    BannersBannerType.mcbanner,
                    BannersBannerType.internal);

    /**
     * Строка переданная в качестве ссылки является ссылкой
     */
    @SuppressWarnings("WeakerAccess")
    public static Validator<String, Defect> imageUrlIsValid() {
        return imageUrl -> {
            ItemValidationBuilder<String, Defect> vb = ItemValidationBuilder.of(imageUrl);
            vb.check(notNull())
                    .check(notBlank())
                    .check(validHref(), When.isValid());
            return vb.getResult();

        };
    }

    /**
     * MimeType изображения поддерживается
     */
    public static <T> Constraint<T, Defect> imageMimeTypeIsSupported(@Nullable String mimeType) {
        return fromPredicate(value -> mimeType == null || isSupportedImageMimeType(mimeType),
                ImageDefects.imageMimeTypeIsNotSupported());
    }

    /**
     * Изображение статично
     */
    public static <T> Constraint<T, Defect> imageDoesNotContainsAnimation(Integer framesNumber) {
        return fromPredicate(value -> framesNumber == 1,
                ImageDefects.imageContainsAnimation());
    }

    /**
     * Размер файла изображения не превышает допустимого
     */
    public static <T> Constraint<T, Defect> imageFileSizeIsValid(byte[] imageData, int maxSize) {
        return fromPredicate(value -> imageData.length <= maxSize,
                ImageDefects.imageFileSizeGreaterThanMax());
    }

    public static <T> Constraint<T, Defect> imageFileSizeIsValid(int size, int maxSize) {
        return fromPredicate(value -> size <= maxSize,
                ImageDefects.imageFileSizeGreaterThanMax());
    }

    /**
     * Линейные размеры изображения не превышают максимальную длину)
     */
    public static <T> Constraint<T, Defect> imageSizesNotGreaterThanMax(ImageSize imageSize) {
        return fromPredicate(index -> imageSize.getWidth() <= MAX_IMAGE_LINEAR_SIZE_IN_PIXEL
                        && imageSize.getHeight() <= MAX_IMAGE_LINEAR_SIZE_IN_PIXEL,
                ImageDefects.imageSizesGreaterThanMax(MAX_IMAGE_LINEAR_SIZE_IN_PIXEL));
    }

    public static <T> Constraint<T, Defect> imageFormatIsAllowed(ImageFileFormat format,
                                                                 Set<ImageFileFormat> allowedFormats) {
        return fromPredicate(index -> allowedFormats.contains(format),
                ImageDefects.imageFileFormatIsNotAllowed(allowedFormats));
    }
}
