package ru.yandex.direct.core.entity.image.service.validation;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.image.service.ImageUtils.getMimeType;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageMimeTypeIsSupported;

@Service
@ParametersAreNonnullByDefault
public class SaveImageValidationService {
    public void validateMimeType(ValidationResult<List<Integer>, Defect> validationResult,
                                 Map<Integer, byte[]> imageFileById) {
        ListValidationBuilder<Integer, Defect> vb = new ListValidationBuilder<>(validationResult);
        vb.checkEachBy(imageFileId -> {
            if (imageFileById.containsKey(imageFileId)) {
                return SaveImageValidationService.<Integer>imageHasValidMimeType(imageFileById.get(imageFileId))
                        .apply(imageFileId);
            }
            return ValidationResult.success(imageFileId);
        });
    }

    public <T> void validateMimeType(ValidationResult<T, Defect> validationResult, byte[] imageData) {
        ItemValidationBuilder<T, Defect> vb = new ItemValidationBuilder<>(validationResult);
        vb.checkBy(imageHasValidMimeType(imageData));
    }

    /**
     * Mime-type файла с изображением поддерживается
     */
    public static <T> Validator<T, Defect> imageHasValidMimeType(byte[] imageData) {
        return url -> {
            String mimeType;
            mimeType = getMimeType(imageData);
            ItemValidationBuilder<T, Defect> vb = ItemValidationBuilder.of(url);
            vb.check(imageMimeTypeIsSupported(mimeType));
            return vb.getResult();
        };
    }
}
