package ru.yandex.direct.core.entity.image.service.validation.type;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.core.entity.image.container.ImageMetaInformation;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.ImageValidationContainer;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImage;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_INTERNAL_BANNER;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFileSizeIsValid;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFormatIsAllowed;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageSizesNotGreaterThanMax;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;

@Component
@ParametersAreNonnullByDefault
public class ImageSaveValidationForInternalBannerAdSupport implements ImageSaveValidationSupport {
    private final static Set<ImageFileFormat> ALLOWED_FORMATS =
            Set.of(ImageFileFormat.PNG, ImageFileFormat.GIF, ImageFileFormat.JPEG, ImageFileFormat.SVG);

    @Override
    public BannerImageType getBannerImageType() {
        return BannerImageType.BANNER_INTERNAL;
    }

    @Override
    public ValidationResult<List<Integer>, Defect> validate(
            List<ImageMetaInformation> imageMetaInformationList,
            ValidationResult<List<Integer>, Defect> vr,
            Map<Integer, Integer> imageIdToIndexOfFetchedImage) {
        ListValidationBuilder<Integer, Defect> lb = new ListValidationBuilder<>(vr);
        return lb
                .check(notEmptyCollection(), When.isValid())
                .checkEachBy(
                        index -> validateImageMetaInformation(index, imageMetaInformationList.get(
                                imageIdToIndexOfFetchedImage.get(index))),
                        When.isValid())
                .getResult();
    }

    public <T> ValidationResult<T, Defect> validateImageMetaInformation(T value,
                                                                        ImageMetaInformation imageMetaInformation) {
        return ModelItemValidationBuilder.<T, Defect>of(value)
                .check(imageSizesNotGreaterThanMax(imageMetaInformation.getSize()))
                .check(imageFileSizeIsValid(imageMetaInformation.getImageFileSize(),
                        MAX_IMAGE_FILE_SIZE_FOR_INTERNAL_BANNER))
                .check(imageFormatIsAllowed(imageMetaInformation.getFormat(), ALLOWED_FORMATS))
                .getResult();
    }

    @Override
    public <T> ValidationResult<T, Defect> validateBeforeSave(T value,
                                                              ImageValidationContainer validationContainer) {
        var resourceInfo = validationContainer.getResourceInfo();
        var imageFormat = validationContainer.getBannerImageFormat();
        if (resourceInfo == null || CollectionUtils.isEmpty(resourceInfo.getValueRestrictions())) {
            return ValidationResult.success(value);
        }

        var vb = ModelItemValidationBuilder.<T, Defect>of(value);

        StreamEx.of(resourceInfo.getValueRestrictions())
                .filter(restriction -> restriction instanceof RestrictionImage)
                .forEach(restriction -> checkRestriction(vb, imageFormat, (RestrictionImage) restriction));

        return vb.getResult();
    }

    private <T> void checkRestriction(ItemValidationBuilder<T, Defect> vb,
                                      BannerImageFormat input,
                                      RestrictionImage restriction) {
        var defect = restriction.check(input);
        if (restriction.isStrict()) {
            vb.check(i -> defect);
        } else {
            vb.weakCheck(i -> defect);
        }
    }
}
