package ru.yandex.direct.core.entity.image.service.validation.type;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.core.entity.image.container.ImageMetaInformation;
import ru.yandex.direct.core.entity.image.service.validation.ImageDefects;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGES_PER_REQUEST_FOR_LOGO;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_LOGO;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.isSizeAvailableForLogo;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageDoesNotContainsAnimation;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFileSizeIsValid;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFormatIsAllowed;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageSizesNotGreaterThanMax;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;

@Component
@ParametersAreNonnullByDefault
public class ImageSaveValidationForLogoSupport implements ImageSaveValidationSupport {
    private final static Set<ImageFileFormat> ALLOWED_FORMATS =
            Set.of(ImageFileFormat.PNG, ImageFileFormat.GIF, ImageFileFormat.JPEG);

    @Override
    public BannerImageType getBannerImageType() {
        return BannerImageType.ASSET_LOGO;
    }

    @Override
    public ValidationResult<List<Integer>, Defect> validate(
            List<ImageMetaInformation> imageMetaInformationList,
            ValidationResult<List<Integer>, Defect> vr,
            Map<Integer, Integer> imageIdToIndexOfFetchedImage) {
        ListValidationBuilder<Integer, Defect> lb = new ListValidationBuilder<>(vr);
        return lb
                .check(notEmptyCollection(), When.isValid())
                .check(maxListSize(MAX_IMAGES_PER_REQUEST_FOR_LOGO), When.isValid())
                .checkEachBy(
                        index -> validateImageMetaInformation(index, imageMetaInformationList.get(
                                imageIdToIndexOfFetchedImage.get(index))),
                        When.isValid())
                .getResult();
    }

    public <T> ValidationResult<T, Defect> validateImageMetaInformation(T value,
                                                                        ImageMetaInformation imageMetaInformation) {
        ItemValidationBuilder<T, Defect> vb = ModelItemValidationBuilder.of(value);

        vb.check(imageSizeIsValid(imageMetaInformation.getSize()))
                .check(imageSizesNotGreaterThanMax(imageMetaInformation.getSize()))
                .check(imageFileSizeIsValid(imageMetaInformation.getImageFileSize(), MAX_IMAGE_FILE_SIZE_FOR_LOGO))
                .check(imageDoesNotContainsAnimation(imageMetaInformation.getFramesNumber()))
                .check(imageFormatIsAllowed(imageMetaInformation.getFormat(), ALLOWED_FORMATS));

        return vb.getResult();
    }

    /**
     * Размер изображения валидный
     */
    private static <T> Constraint<T, Defect> imageSizeIsValid(ImageSize imageSize) {
        return fromPredicate(index -> isSizeAvailableForLogo(imageSize.getWidth(), imageSize.getHeight()),
                ImageDefects.imageSizeIsNotAllowed());
    }
}
