package ru.yandex.direct.core.entity.image.service.validation.type;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.core.entity.image.container.ImageMetaInformation;
import ru.yandex.direct.core.entity.image.service.validation.ImageDefects;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGES_PER_REQUEST_FOR_MULTICARD;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_MULTICARD;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.isSizeAvailableForNewImages;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageDoesNotContainsAnimation;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFileSizeIsValid;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFormatIsAllowed;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageSizesNotGreaterThanMax;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;

@Component
@ParametersAreNonnullByDefault
public class ImageSaveValidationForMulticardSupport implements ImageSaveValidationSupport {
    private final static Set<ImageFileFormat> ALLOWED_FORMATS =
            Set.of(ImageFileFormat.PNG, ImageFileFormat.GIF, ImageFileFormat.JPEG);

    @Override
    public BannerImageType getBannerImageType() {
        return BannerImageType.ASSET_MULTICARD;
    }

    @Override
    public ValidationResult<List<Integer>, Defect> validate(
            List<ImageMetaInformation> imageMetaInformationList,
            ValidationResult<List<Integer>, Defect> vr,
            Map<Integer, Integer> imageIdToIndexOfFetchedImage) {
        ListValidationBuilder<Integer, Defect> lb = new ListValidationBuilder<>(vr);
        Set<Integer> indexes = new HashSet<>();
        return lb
                .check(notEmptyCollection(), When.isValid())
                .checkEachBy(
                        index -> validateImageMetaInformation(index, imageMetaInformationList.get(
                                imageIdToIndexOfFetchedImage.get(index))),
                        When.isValid())
                .checkEachBy(
                        index -> validateImageOrderNumber(index, indexes),
                        When.isValid())
                .getResult();
    }

    /**
     * Возвращает ошибку валидации после достижения лимита валидных элементов
     * @param index индекс валидируемого
     * @param indexes индексы провалидированных ранее элементов
     * @return результат валидации
     */
    public ValidationResult<Integer, Defect> validateImageOrderNumber(Integer index, Set<Integer> indexes) {
        ItemValidationBuilder<Integer, Defect> vb = ModelItemValidationBuilder.of(index);
        return vb
                .checkByFunction(ind ->
                        indexes.add(index) && indexes.size() > MAX_IMAGES_PER_REQUEST_FOR_MULTICARD
                                ? CollectionDefects.maxCollectionSize(MAX_IMAGES_PER_REQUEST_FOR_MULTICARD)
                                : null
                )
                .getResult();
    }

    public <T> ValidationResult<T, Defect> validateImageMetaInformation(T value,
                                                                        ImageMetaInformation imageMetaInformation) {
        ItemValidationBuilder<T, Defect> vb = ModelItemValidationBuilder.of(value);

        vb.check(imageSizeIsValid(imageMetaInformation.getSize()))
                .check(imageSizesNotGreaterThanMax(imageMetaInformation.getSize()))
                .check(imageFileSizeIsValid(imageMetaInformation.getImageFileSize(),
                        MAX_IMAGE_FILE_SIZE_FOR_MULTICARD))
                .check(imageDoesNotContainsAnimation(imageMetaInformation.getFramesNumber()))
                .check(imageFormatIsAllowed(imageMetaInformation.getFormat(), ALLOWED_FORMATS));

        return vb.getResult();
    }

    /**
     * Размер изображения валидный
     */
    private static <T> Constraint<T, Defect> imageSizeIsValid(ImageSize imageSize) {
        return fromPredicate(index -> isSizeAvailableForNewImages(imageSize.getWidth(), imageSize.getHeight()),
                ImageDefects.imageSizeIsNotAllowed());
    }
}
