package ru.yandex.direct.core.entity.image.service.validation.type;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.feature.service.FeatureHelper;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.ImageFileFormat;
import ru.yandex.direct.core.entity.image.container.ImageMetaInformation;
import ru.yandex.direct.core.entity.image.service.validation.ImageDefects;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.ALLOWED_SIZES_FOR_AD_IMAGE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.ALLOWED_SIZES_FOR_AD_IMAGE_ORIGINAL;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGES_PER_REQUEST_FOR_IMAGE_AD;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_TEXT_IMAGE_BANNER;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.NEW_MAX_IMAGE_FILE_SIZE_FOR_TEXT_IMAGE_BANNER;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFileSizeIsValid;
import static ru.yandex.direct.core.entity.image.service.validation.ImageConstraints.imageFormatIsAllowed;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;

@Component
@ParametersAreNonnullByDefault
public class ImageSaveValidationForTextImageAdSupport implements ImageSaveValidationSupport {
    private final static Set<ImageFileFormat> ALLOWED_FORMATS =
            Set.of(ImageFileFormat.PNG, ImageFileFormat.GIF, ImageFileFormat.JPEG);
    private final static int DEFAULT_INACCURACY = 3;
    private final static int DEFAULT_MAX_RATIO = 4;
    private final FeatureService featureService;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public ImageSaveValidationForTextImageAdSupport(FeatureService featureService, PpcPropertiesSupport ppcPropertiesSupport) {
        this.featureService = featureService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public BannerImageType getBannerImageType() {
        return BannerImageType.BANNER_IMAGE_AD;
    }

    @Override
    public ValidationResult<List<Integer>, Defect> validate(
            List<ImageMetaInformation> imageMetaInformationList,
            ValidationResult<List<Integer>, Defect> vr,
            Map<Integer, Integer> imageIdToIndexOfFetchedImage) {
        ListValidationBuilder<Integer, Defect> lb = new ListValidationBuilder<>(vr);
        return lb
                .check(notEmptyCollection(), When.isValid())
                .check(maxListSize(MAX_IMAGES_PER_REQUEST_FOR_IMAGE_AD), When.isValid())
                .checkEachBy(
                        index -> validateImageMetaInformation(index, imageMetaInformationList.get(
                                imageIdToIndexOfFetchedImage.get(index))),
                        When.isValid())
                .getResult();
    }

    public <T> ValidationResult<T, Defect> validateImageMetaInformation(T value,
                                                                        ImageMetaInformation imageMetaInformation) {
        ItemValidationBuilder<T, Defect> vb = ModelItemValidationBuilder.of(value);

        var allowProportionallyLargerImage = FeatureHelper.feature(FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES).enabled();

        vb
                .check(largeImageSizeIsValid(imageMetaInformation.getSize()), When.isTrue(allowProportionallyLargerImage))
                .check(imageSizeIsValid(imageMetaInformation.getSize()), When.isFalse(allowProportionallyLargerImage))
                .check(imageFileSizeIsValid(imageMetaInformation.getImageFileSize(),
                        allowProportionallyLargerImage
                                ? NEW_MAX_IMAGE_FILE_SIZE_FOR_TEXT_IMAGE_BANNER
                                : MAX_IMAGE_FILE_SIZE_FOR_TEXT_IMAGE_BANNER))
                .check(imageFormatIsAllowed(imageMetaInformation.getFormat(), ALLOWED_FORMATS));

        return vb.getResult();
    }

    /**
     * Размер изображения валидный
     */
    private  <T> Constraint<T, Defect> largeImageSizeIsValid(ImageSize imageSize) {
            return fromPredicate(
                    value -> isProportionallyLarger(imageSize),
                    ImageDefects.imageSizeIsNotAllowed());
    }

    /**
     * Размер изображения валидный
     */
    private static <T> Constraint<T, Defect> imageSizeIsValid(ImageSize imageSize) {
        return fromPredicate(
                value -> ALLOWED_SIZES_FOR_AD_IMAGE.contains(imageSize),
                ImageDefects.imageSizeIsNotAllowed());
    }

    public  boolean isProportionallyLarger(ImageSize imageSize) {
        var inaccuracy = ppcPropertiesSupport.get(
                PpcPropertyNames.INACCURACY_IN_PERCENTS_FOR_PROPORTIONALLY_LARGER_IMAGES,
                Duration.ofMinutes(5)
        ).getOrDefault(DEFAULT_INACCURACY);
        inaccuracy = getValueInLimits(inaccuracy, 0, 100);

        var maxRatio = ppcPropertiesSupport.get(
                PpcPropertyNames.MAX_RATIO_FOR_PROPORTIONALLY_LARGER_IMAGES,
                Duration.ofMinutes(5)
        ).getOrDefault(DEFAULT_MAX_RATIO);
        maxRatio = getValueInLimits(maxRatio, 1, 100);

        for (var baseImageSize : ALLOWED_SIZES_FOR_AD_IMAGE_ORIGINAL) {
            var ratio = Math.round(imageSize.getHeight().doubleValue() / baseImageSize.getHeight());
            if (ratio == 0) {
                continue;
            }
            //проверяем, что наше изображение в натуральное число раз больше разрешенных, с погрешностью
            if (ratio <= maxRatio &&
                    Math.abs(baseImageSize.getHeight() * ratio - imageSize.getHeight()) <= baseImageSize.getHeight() * inaccuracy.doubleValue() / 100 &&
                    Math.abs(baseImageSize.getWidth() * ratio - imageSize.getWidth()) <= baseImageSize.getWidth() * inaccuracy.doubleValue() / 100) {

                imageSize.setHeight(baseImageSize.getHeight());
                imageSize.setWidth(baseImageSize.getWidth());
                return true;
            }
        }
        return false;
    }

    private static int getValueInLimits(int value, int min, int max) {
        value = Math.max(value, min);
        return Math.min(value, max);
    }
}
