package ru.yandex.direct.core.entity.internalads.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.ArrayUtils;

import ru.yandex.direct.core.entity.internalads.restriction.Restriction;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class InternalTemplateOverrides {

    private String resourceRestrictionsErrorMessage;
    private final List<ResourceRestrictionOverrideInfo> resourceRestrictionOverrides;

    private final Map<Long, List<Restriction>> valueRestrictionOverrides;
    private final Map<Long, List<ResourceChoice>> resourceChoices;
    private final Set<Long> hiddenResources;

    public InternalTemplateOverrides() {
        this.resourceRestrictionsErrorMessage = null;
        this.resourceRestrictionOverrides = new ArrayList<>();
        this.valueRestrictionOverrides = new HashMap<>();
        this.resourceChoices = new HashMap<>();
        this.hiddenResources = new HashSet<>();
    }

    public InternalTemplateOverrides requiredVariantErrorMessage(String errorMessage) {
        this.resourceRestrictionsErrorMessage = errorMessage;
        return this;
    }

    public InternalTemplateOverrides requiredVariant(Set<Long> required, Set<Long> absent) {
        resourceRestrictionOverrides.add(new ResourceRestrictionOverrideInfo(required, absent));
        return this;
    }

    public InternalTemplateOverrides valueRestriction(long resourceId, List<Restriction> restrictions) {
        valueRestrictionOverrides.put(resourceId, restrictions);
        return this;
    }

    public InternalTemplateOverrides addResourceChoices(long resourceId, List<String> choices) {
        resourceChoices.put(resourceId, mapList(choices, ResourceChoice::from));
        return this;
    }

    public InternalTemplateOverrides addHiddenResources(Long... resourceIds) {
        checkArgument(ArrayUtils.isNotEmpty(resourceIds), "hidden resourceIds must not be empty");
        hiddenResources.addAll(List.of(resourceIds));
        return this;
    }

    /**
     * Применить переопределения к переданному объекту с информацией о шаблоне и его ресурсах.
     * <p>
     * Внимание! Предполагается, что в параметре templateInfo присутствует только один вариант ограничения
     * присутствия ресурсов resourceRestriction
     * <p>
     * Внимание! Метод меняет состояние переданного параметра!
     *
     * @param templateInfo объект с информацией о шаблоне и его ресурсах
     */
    public InternalTemplateInfo applyTo(InternalTemplateInfo templateInfo) {
        checkArgument(templateInfo.getResourceRestrictions().size() == 1,
                "function works with the only resourceRestriction into templateInfo");

        if (resourceRestrictionOverrides.isEmpty() && valueRestrictionOverrides.isEmpty() && resourceChoices.isEmpty()
                && hiddenResources.isEmpty()) {
            return templateInfo;
        }

        if (!resourceRestrictionOverrides.isEmpty()) {
            ResourceRestriction theOnlyResourceRestriction = templateInfo.getResourceRestrictions().get(0);
            List<ResourceRestriction> newResourceRestrictions = StreamEx.of(resourceRestrictionOverrides)
                    .map(overrideInfo -> mergeResourceRestrictions(theOnlyResourceRestriction, overrideInfo))
                    .toImmutableList();
            templateInfo.setResourceRestrictions(newResourceRestrictions);
            if (resourceRestrictionsErrorMessage != null) {
                templateInfo.setResourceRestrictionsErrorMessage(resourceRestrictionsErrorMessage);
            }
        }

        for (ResourceInfo resource : templateInfo.getResources()) {
            List<ResourceChoice> choices = resourceChoices.get(resource.getId());
            if (choices != null && !choices.isEmpty()) {
                resource.setChoices(choices);
            }
            List<Restriction> valueRestrictions = valueRestrictionOverrides.get(resource.getId());
            if (valueRestrictions != null && !valueRestrictions.isEmpty()) {
                resource.setValueRestrictions(valueRestrictions);
            }
            resource.setHidden(hiddenResources.contains(resource.getId()));
        }
        return templateInfo;
    }

    private static ResourceRestriction mergeResourceRestrictions(ResourceRestriction resourceRestriction,
                                                                 ResourceRestrictionOverrideInfo overrideInfo) {
        Set<Long> absent = Sets.union(resourceRestriction.getAbsent(), overrideInfo.absentResourceIds);
        Set<Long> required = Sets.difference(Sets.union(resourceRestriction.getRequired(),
                overrideInfo.requiredResourceIds), absent);
        return new ResourceRestriction()
                .withRequired(required)
                .withAbsent(absent);
    }

    private static class ResourceRestrictionOverrideInfo {
        private final Set<Long> requiredResourceIds;
        private final Set<Long> absentResourceIds;

        private ResourceRestrictionOverrideInfo(Set<Long> requiredResourceIds, Set<Long> absentResourceIds) {
            this.requiredResourceIds = requiredResourceIds;
            this.absentResourceIds = absentResourceIds;
        }
    }
}
