package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.DirectTemplatePlace.DIRECT_TEMPLATE_PLACE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class DirectTemplatePlaceRepository {

    private static final JooqMapperWithSupplier<TemplatePlace> MAPPER =
            JooqMapperWithSupplierBuilder.builder(TemplatePlace::new)
                    .map(property(TemplatePlace.TEMPLATE_ID, DIRECT_TEMPLATE_PLACE.DIRECT_TEMPLATE_ID))
                    .map(property(TemplatePlace.PLACE_ID, DIRECT_TEMPLATE_PLACE.PLACE_ID))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public DirectTemplatePlaceRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить все записи таблицы {@code direct_template_place}
     *
     * @return список связок площадки и шаблона
     */
    public List<TemplatePlace> getAll() {
        return get(DSL.trueCondition());
    }

    /**
     * Получить записи по placeId из таблицы {@code direct_template_place}
     *
     * @param placeIds — ID площадок
     * @return список связок площадки и шаблона
     */
    public List<TemplatePlace> getByPlaceIds(Collection<Long> placeIds) {
        return get(DIRECT_TEMPLATE_PLACE.PLACE_ID.in(placeIds));
    }

    /**
     * Получить записи по directTemplateId из таблицы {@code direct_template_place}
     *
     * @param directTemplateId — ID шаблона
     * @return список связок площадки и шаблона
     */
    public List<TemplatePlace> getByTemplateId(Long directTemplateId) {
        return get(DIRECT_TEMPLATE_PLACE.DIRECT_TEMPLATE_ID.eq(directTemplateId));
    }

    /**
     * Получить записи, удовлетворяющие переданному условию, из таблицы {@code direct_template_place}
     */
    private List<TemplatePlace> get(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(DIRECT_TEMPLATE_PLACE)
                .where(condition)
                .fetch(MAPPER::fromDb);
    }

    /**
     * Для каких плейсов есть шаблоны?
     *
     * @return уникальные ID площадок
     */
    public List<Long> getPlaces() {
        return dslContextProvider.ppcdict()
                .selectDistinct(DIRECT_TEMPLATE_PLACE.PLACE_ID)
                .from(DIRECT_TEMPLATE_PLACE)
                .fetch(DIRECT_TEMPLATE_PLACE.PLACE_ID);
    }

    /**
     * Получить записи по directTemplateId из таблицы {@code direct_template_place}.
     *
     * @param directTemplateIds — ID шаблонов
     * @return словарь id шаблона -> список площадок
     */
    public Map<Long, List<TemplatePlace>> getByTemplateIds(Collection<Long> directTemplateIds) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(DIRECT_TEMPLATE_PLACE)
                .where(DIRECT_TEMPLATE_PLACE.DIRECT_TEMPLATE_ID.in(directTemplateIds))
                .fetchGroups(DIRECT_TEMPLATE_PLACE.DIRECT_TEMPLATE_ID, MAPPER::fromDb);
    }

    /**
     * Добавить записи
     *
     * @param addRecords — связки типа TemplatePlace, которые нужно добавить
     */
    public void add(Collection<TemplatePlace> addRecords) {
        if (addRecords.isEmpty()) {
            return;
        }

        new InsertHelper<>(dslContextProvider.ppcdict(), DIRECT_TEMPLATE_PLACE)
                .addAll(MAPPER, addRecords)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    /**
     * Удалить записи
     *
     * @param deleteRecords — связки типа TemplatePlace, которые нужно удалить
     */
    public void delete(Collection<TemplatePlace> deleteRecords) {
        if (deleteRecords.isEmpty()) {
            return;
        }

        var step = dslContextProvider.ppcdict()
                .deleteFrom(DIRECT_TEMPLATE_PLACE)
                .where(DSL.falseCondition());
        deleteRecords.forEach(tp -> step.or(DIRECT_TEMPLATE_PLACE.DIRECT_TEMPLATE_ID.eq(tp.getTemplateId())
                .and(DIRECT_TEMPLATE_PLACE.PLACE_ID.eq(tp.getPlaceId()))));

        step.execute();
    }

}
