package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.DirectTemplate;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateState;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppcdict.tables.DirectTemplate.DIRECT_TEMPLATE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class DirectTemplateRepository {
    private static final JooqMapperWithSupplier<DirectTemplate> MAPPER =
            JooqMapperWithSupplierBuilder.builder(DirectTemplate::new)
                    .map(property(DirectTemplate.DIRECT_TEMPLATE_ID, DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID))
                    .map(property(DirectTemplate.FORMAT_NAME, DIRECT_TEMPLATE.FORMAT_NAME))
                    .map(convertibleProperty(DirectTemplate.STATE, DIRECT_TEMPLATE.STATE,
                            DirectTemplateRepository::stateFromDb, DirectTemplateRepository::stateToDb))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    DirectTemplateRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить свойства всех существующих шаблонов.
     * Предполагается использовать только во внутренних инструментах.
     *
     * @return словарь id шаблона -> свойства шаблона
     */
    public Map<Long, DirectTemplate> getAll() {
        return get(DSL.trueCondition());
    }

    /**
     * Получить свойства шаблонов по их id
     *
     * @param directTemplateIds — коллекция id шаблонов
     * @return словарь из id шаблонов в свойства шаблонов типа DirectTemplate
     */
    public Map<Long, DirectTemplate> get(Collection<Long> directTemplateIds) {
        if (directTemplateIds.isEmpty()) {
            return Map.of();
        }
        return get(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID.in(directTemplateIds));
    }

    private Map<Long, DirectTemplate> get(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(DIRECT_TEMPLATE)
                .where(condition)
                .fetchMap(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID, MAPPER::fromDb);
    }

    /**
     * Завести новый шаблон.
     * Предполагается использовать только во внутренних инструментах
     *
     * @param directTemplate — формат и состояние шаблона
     * @return новый id шаблона
     */
    public Long add(DirectTemplate directTemplate) {
        return dslContextProvider.ppcdict()
                .insertInto(DIRECT_TEMPLATE)
                .columns(DIRECT_TEMPLATE.FORMAT_NAME, DIRECT_TEMPLATE.STATE)
                .values(directTemplate.getFormatName(), stateToDb(directTemplate.getState()))
                .returningResult(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID)
                .fetchOne()
                .value1();
    }

    /**
     * Завести старый шаблон, т. е. обозначить, что он мигрируется на новый формат.
     * Предполагается использовать только во внутренних инструментах
     *
     * @param directTemplate — id, формат и состояние шаблона
     */
    public void addOldTemplate(DirectTemplate directTemplate) {
        dslContextProvider.ppcdict()
                .insertInto(DIRECT_TEMPLATE)
                .columns(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID, DIRECT_TEMPLATE.FORMAT_NAME, DIRECT_TEMPLATE.STATE)
                .values(directTemplate.getDirectTemplateId(),
                        directTemplate.getFormatName(),
                        stateToDb(directTemplate.getState()))
                .execute();
    }

    /**
     * Удалить шаблон
     *
     * @param directTemplateId — id шаблона, который нужно удалить
     */
    public void delete(long directTemplateId) {
        dslContextProvider.ppcdict()
                .deleteFrom(DIRECT_TEMPLATE)
                .where(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID.eq(directTemplateId))
                .execute();
    }

    /**
     * Обновить название формата для единого шаблона.
     * Предполагается использовать только во внутренних инструментах.
     *
     * @param directTemplateId — id шаблона
     * @param formatName       — новое название формата
     */
    public void setFormatName(Long directTemplateId, String formatName) {
        dslContextProvider.ppcdict()
                .update(DIRECT_TEMPLATE)
                .set(DIRECT_TEMPLATE.FORMAT_NAME, formatName)
                .where(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID.eq(directTemplateId))
                .execute();
    }

    /**
     * Обновить состояние (способ отправки в БК) единого шаблона
     *
     * @param directTemplateId — id шаблона
     * @param state            — новое состояние
     */
    public void setState(Long directTemplateId, DirectTemplateState state) {
        dslContextProvider.ppcdict()
                .update(DIRECT_TEMPLATE)
                .set(DIRECT_TEMPLATE.STATE, stateToDb(state))
                .where(DIRECT_TEMPLATE.DIRECT_TEMPLATE_ID.eq(directTemplateId))
                .execute();
    }

    private static ru.yandex.direct.dbschema.ppcdict.enums.DirectTemplateState stateToDb(DirectTemplateState t) {
        switch (t) {
            case UNIFIED:
                return ru.yandex.direct.dbschema.ppcdict.enums.DirectTemplateState.unified;
            case TRANSITIONAL:
                return ru.yandex.direct.dbschema.ppcdict.enums.DirectTemplateState.transitional;
            default:
                return ru.yandex.direct.dbschema.ppcdict.enums.DirectTemplateState.default_;
        }
    }

    private static DirectTemplateState stateFromDb(ru.yandex.direct.dbschema.ppcdict.enums.DirectTemplateState r) {
        switch (r) {
            case unified:
                return DirectTemplateState.UNIFIED;
            case transitional:
                return DirectTemplateState.TRANSITIONAL;
            default:
                return DirectTemplateState.DEFAULT;
        }
    }
}
