package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResourceOption;
import ru.yandex.direct.dbschema.ppcdict.tables.records.DirectTemplateResourceRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppcdict.tables.DirectTemplateResource.DIRECT_TEMPLATE_RESOURCE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class DirectTemplateResourceRepository {

    private static final JooqMapperWithSupplier<DirectTemplateResource> MAPPER =
            JooqMapperWithSupplierBuilder.builder(DirectTemplateResource::new)
                    .map(property(DirectTemplateResource.DIRECT_TEMPLATE_RESOURCE_ID,
                            DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_RESOURCE_ID))
                    .map(property(DirectTemplateResource.DIRECT_TEMPLATE_ID,
                            DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_ID))
                    .map(property(DirectTemplateResource.RESOURCE_NO, DIRECT_TEMPLATE_RESOURCE.RESOURCE_NO))
                    .map(property(DirectTemplateResource.UNIFIED_RESOURCE_NO,
                            DIRECT_TEMPLATE_RESOURCE.UNIFIED_RESOURCE_NO))
                    .map(property(DirectTemplateResource.UNIFIED_TEMPLATE_RESOURCE_ID,
                            DIRECT_TEMPLATE_RESOURCE.UNIFIED_TEMPLATE_RESOURCE_ID))
                    .map(convertibleEnumSet(DirectTemplateResource.OPTIONS, DIRECT_TEMPLATE_RESOURCE.OPTIONS,
                            DirectTemplateResourceOption::fromTypedValue,
                            DirectTemplateResourceOption::getTypedValue))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public DirectTemplateResourceRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить записи по id из таблицы {@code direct_template_resource}
     */
    public List<DirectTemplateResource> getByIds(Collection<Long> directTemplateResourceIds) {
        return get(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_RESOURCE_ID.in(directTemplateResourceIds));
    }

    /**
     * Получить записи по templateId из таблицы {@code direct_template_resource}
     *
     * @param directTemplateIds — ID шаблонов
     * @return список ресурсов
     */
    public List<DirectTemplateResource> getByTemplateIds(Collection<Long> directTemplateIds) {
        return get(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_ID.in(directTemplateIds));
    }

    /**
     * Получить записи, удовлетворяющие переданному условию, из таблицы {@code direct_template_resource}
     */
    private List<DirectTemplateResource> get(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(DIRECT_TEMPLATE_RESOURCE)
                .where(condition)
                .fetch(MAPPER::fromDb);
    }

    /**
     * Получить записи по directTemplateId из таблицы {@code direct_template_resource}.
     *
     * @param directTemplateIds — ID шаблонов
     * @return словарь id шаблона -> список ресурсов
     */
    public Map<Long, List<DirectTemplateResource>> getMapByTemplateIds(Collection<Long> directTemplateIds) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(DIRECT_TEMPLATE_RESOURCE)
                .where(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_ID.in(directTemplateIds))
                .fetchGroups(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_ID, MAPPER::fromDb);
    }

    public void addOrUpdate(Collection<DirectTemplateResource> addOrUpdateRecords) {
        InsertHelper<DirectTemplateResourceRecord> insertHelper = new InsertHelper<>(
                dslContextProvider.ppcdict(), DIRECT_TEMPLATE_RESOURCE
        ).addAll(MAPPER, addOrUpdateRecords);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_RESOURCE_ID,
                            MySQLDSL.values(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_RESOURCE_ID))
                    .set(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_ID,
                            MySQLDSL.values(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_ID))
                    .set(DIRECT_TEMPLATE_RESOURCE.RESOURCE_NO, MySQLDSL.values(DIRECT_TEMPLATE_RESOURCE.RESOURCE_NO))
                    .set(DIRECT_TEMPLATE_RESOURCE.UNIFIED_RESOURCE_NO,
                            MySQLDSL.values(DIRECT_TEMPLATE_RESOURCE.UNIFIED_RESOURCE_NO))
                    .set(DIRECT_TEMPLATE_RESOURCE.UNIFIED_TEMPLATE_RESOURCE_ID,
                            MySQLDSL.values(DIRECT_TEMPLATE_RESOURCE.UNIFIED_TEMPLATE_RESOURCE_ID))
                    .set(DIRECT_TEMPLATE_RESOURCE.OPTIONS, MySQLDSL.values(DIRECT_TEMPLATE_RESOURCE.OPTIONS));
        }

        insertHelper.executeIfRecordsAdded();
    }

    public void delete(Collection<Long> directTemplateResourceIds) {
        if (directTemplateResourceIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppcdict()
                .deleteFrom(DIRECT_TEMPLATE_RESOURCE)
                .where(DIRECT_TEMPLATE_RESOURCE.DIRECT_TEMPLATE_RESOURCE_ID.in(directTemplateResourceIds))
                .execute();
    }

}
