package ru.yandex.direct.core.entity.internalads.repository;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductOption;
import ru.yandex.direct.dbschema.ppc.tables.records.InternalAdProductsRecord;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppc.Tables.INTERNAL_AD_PRODUCTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@ParametersAreNonnullByDefault
@Repository
public class InternalAdsProductRepository {

    private static final JooqMapperWithSupplier<InternalAdsProduct> PRODUCT_MAPPER =
            JooqMapperWithSupplierBuilder.builder(InternalAdsProduct::new)
                    .map(clientIdProperty(InternalAdsProduct.CLIENT_ID, INTERNAL_AD_PRODUCTS.CLIENT_ID))
                    .map(property(InternalAdsProduct.NAME, INTERNAL_AD_PRODUCTS.PRODUCT_NAME))
                    .map(property(InternalAdsProduct.DESCRIPTION, INTERNAL_AD_PRODUCTS.PRODUCT_DESCRIPTION))
                    .map(convertibleEnumSet(InternalAdsProduct.OPTIONS, INTERNAL_AD_PRODUCTS.OPTIONS,
                            InternalAdsProductOption::fromTypedValue,
                            InternalAdsProductOption::getTypedValue))
                    .build();

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public InternalAdsProductRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public int createProduct(int shard, InternalAdsProduct product) {
        DSLContext ppcdict = dslContextProvider.ppcdict();
        String productName = product.getName();
        String lockName = "create_iap:" + productName;
        try {
            Integer lockResult = ppcdict.select(SqlUtils.mysqlGetLock(lockName, Duration.ofSeconds(30)))
                    .fetchOne().component1();
            checkState(Objects.equals(lockResult, 1), "failed to get lock");

            List<ClientId> clientIdsOfProductsWithSameName = getClientIdsOfProductsByName(productName);

            if (!clientIdsOfProductsWithSameName.isEmpty()) {
                throw new IllegalStateException(String.format("product named %s already exists: %s",
                        productName, clientIdsOfProductsWithSameName));
            }

            return new InsertHelper<>(dslContextProvider.ppc(shard), INTERNAL_AD_PRODUCTS)
                    .add(PRODUCT_MAPPER, product)
                    .execute();
        } finally {
            ppcdict.select(SqlUtils.mysqlReleaseLock(lockName)).execute();
        }
    }

    public List<ClientId> getClientIdsOfProductsByName(String productName) {
        return shardHelper.dbShards().stream()
                .flatMap(otherShard -> dslContextProvider.ppc(otherShard)
                        .select(INTERNAL_AD_PRODUCTS.CLIENT_ID)
                        .from(INTERNAL_AD_PRODUCTS)
                        .where(INTERNAL_AD_PRODUCTS.PRODUCT_NAME.eq(productName))
                        .fetchStreamInto(INTERNAL_AD_PRODUCTS)
                        .map(InternalAdProductsRecord::getClientid)
                        .map(ClientId::fromLong))
                .collect(Collectors.toList());
    }

    public List<InternalAdsProduct> getProducts(DSLContext dslContext, Collection<Long> productIds) {
        List<InternalAdsProduct> productList = getProductsNotStrictly(dslContext, productIds);

        checkArgument(productIds.size() == productList.size(), "some of the requested products can't be fetched" +
                "clientIds = %s", productIds.toString());
        return productList;
    }

    public List<InternalAdsProduct> getProductsNotStrictly(DSLContext dslContext, Collection<Long> productIds) {
        return dslContext
                .select(PRODUCT_MAPPER.getFieldsToRead())
                .from(INTERNAL_AD_PRODUCTS)
                .where(INTERNAL_AD_PRODUCTS.CLIENT_ID.in(productIds))
                .fetch(PRODUCT_MAPPER::fromDb);
    }

    public List<InternalAdsProduct> getProductsNotStrictly(int shard, Collection<Long> productIds) {
        return getProductsNotStrictly(dslContextProvider.ppc(shard), productIds);
    }

    public List<InternalAdsProduct> getProducts(int shard, Collection<Long> productIds) {
        return getProducts(dslContextProvider.ppc(shard), productIds);
    }

    public List<InternalAdsProduct> getAllProducts(int shard) {
        return dslContextProvider.ppc(shard)
                .select(PRODUCT_MAPPER.getFieldsToRead())
                .from(INTERNAL_AD_PRODUCTS)
                .fetch(PRODUCT_MAPPER::fromDb);
    }

    public void updateProduct(int shard, InternalAdsProduct internalAdsProduct) {
        dslContextProvider.ppc(shard)
                .update(INTERNAL_AD_PRODUCTS)
                .set(INTERNAL_AD_PRODUCTS.PRODUCT_DESCRIPTION, internalAdsProduct.getDescription())
                .set(INTERNAL_AD_PRODUCTS.OPTIONS, productOptionsToDb(internalAdsProduct.getOptions()))
                .where(INTERNAL_AD_PRODUCTS.CLIENT_ID.eq(internalAdsProduct.getClientId().asLong()))
                .execute();
    }

    private static String productOptionsToDb(Set<InternalAdsProductOption> options) {
        return RepositoryUtils.setToDb(options, InternalAdsProductOption::getTypedValue);
    }

}
