package ru.yandex.direct.core.entity.internalads.repository;

import java.time.LocalDate;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.EnumUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.CryptaExportType;
import ru.yandex.direct.core.entity.internalads.model.CryptaSegment;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtCryptaSegmentsRow;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables.CRYPTASEGMENTS;

@Repository
@ParametersAreNonnullByDefault
public class InternalCryptaSegmentsYtRepository {

    private static final String TABLE_GENERATE_DATE_ATTRIBUTE_NAME = "generate_date";

    private final YtProvider ytProvider;

    @Autowired
    public InternalCryptaSegmentsYtRepository(YtProvider ytProvider) {
        this.ytProvider = ytProvider;
    }

    /**
     * Получить все записи таблицы segments из yt-таблицы, экспортируемой из Крипты
     */
    public List<CryptaSegment> getAll(YtCluster cluster) {
        return ytProvider.getOperator(cluster)
                .readTableAndMap(CRYPTASEGMENTS, new YtCryptaSegmentsRow(), InternalCryptaSegmentsYtRepository::convertFromYt);
    }

    /**
     * Получить дату генерации таблицы {@link YtDbTables#CRYPTASEGMENTS} - таблица создается раз в день
     * дата хранится в атрибуте таблицы {@link #TABLE_GENERATE_DATE_ATTRIBUTE_NAME}
     * Считаем, что атрибут проставлен и имеет корректное значение
     */
    public LocalDate getTableGenerateDate(YtCluster cluster) {
        String generateDate = ytProvider.getOperator(cluster)
                .readTableStringAttribute(CRYPTASEGMENTS, TABLE_GENERATE_DATE_ATTRIBUTE_NAME);
        return LocalDate.parse(generateDate);
    }

    private static CryptaSegment convertFromYt(YtCryptaSegmentsRow row) {
        return new CryptaSegment()
                .withId(row.getId())
                .withParentId(row.getParentId())
                .withName(row.getName())
                .withExportType(toCryptaExportType(row.getExportType()))
                .withExportKeywordId(row.getExportKeywordId())
                .withExportSegmentId(row.getExportSegmentId());
    }

    @Nullable
    private static CryptaExportType toCryptaExportType(@Nullable String exportType) {
        return EnumUtils.getEnum(CryptaExportType.class, exportType);
    }

}
