package ru.yandex.direct.core.entity.internalads.repository;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Timer;

import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;

import org.springframework.stereotype.Component;

import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.selector.ClusterFreshness;
import ru.yandex.direct.ytwrapper.dynamic.selector.WeightBasedClusterChooser;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables.TEMPLATERESOURCE;

/**
 * Выбирает кластер для чтения таблиц, необходимых для внутренней рекламы.
 * <p>
 * Выбор выполняется по времени создания таблицы TemplateResource и по доступности кластера.
 * Если таблицы обновлялись раньше FRESH_DELAY, то кластеры будут в одной группе свежести и,
 * следовательно, иметь одинаковый приоритет для выбора.
 * Если на каком-то кластере окажется более новая таблица, он получит приоритет.
 * Недоступный кластер не будет выбираться -- эта логика заложена в WeightBasedClusterChooser.
 */
@Component
public class InternalYaBsClusterChooser {
    static final String MAX_UNIX_TIME = "max_unix_time";
    private static final Duration FRESH_DELAY = Duration.ofHours(2);
    private static final Duration YT_CLUSTER_REFRESH_INTERVAL = Duration.ofMinutes(5);

    private final YtProvider ytProvider;

    private final Timer timer;
    private final WeightBasedClusterChooser clusterChooser;

    public InternalYaBsClusterChooser(YtProvider ytProvider, InternalYaBsClusterConfig config) {
        this.ytProvider = ytProvider;
        timer = new Timer(getClass().getSimpleName() + "-Times", true);
        clusterChooser = new WeightBasedClusterChooser(
                timer,
                YT_CLUSTER_REFRESH_INTERVAL,
                config.getClusters(),
                this::clusterFreshness);
    }

    @PreDestroy
    public void stopTimer() {
        if (timer != null) {
            timer.cancel();
        }
    }

    private ClusterFreshness clusterFreshness(YtCluster cluster) {
        Long tableUpdateTime = ytProvider.getOperator(cluster)
                .readTableNumericAttribute(TEMPLATERESOURCE, MAX_UNIX_TIME);
        long now = System.currentTimeMillis() / 1000L;
        if (tableUpdateTime + FRESH_DELAY.getSeconds() > now) {
            return ClusterFreshness.FRESH;
        } else {
            return ClusterFreshness.ROTTEN;
        }
    }

    /**
     * Возвращает список доступных кластеров в том порядке, в котором к ним необходимо обращаться
     */
    @Nonnull
    Optional<List<YtCluster>> getClustersOrdered() {
        return clusterChooser.getClustersOrdered();
    }

    /**
     * Возвращает список доступных кластеров в том порядке, в котором к ним необходимо обращаться
     *
     * @throws IllegalStateException если нет доступных кластеров
     */
    @Nonnull
    public List<YtCluster> getAvailableClustersOrdered() {
        return getClustersOrdered().orElseThrow(() -> new IllegalStateException("No any clusters"));
    }

}
