package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DeleteConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.PagePlace;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PagePlaceRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.PagePlace.PAGE_PLACE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class PagePlaceRepository {

    private static final JooqMapperWithSupplier<PagePlace> MAPPER =
            JooqMapperWithSupplierBuilder.builder(PagePlace::new)
                    .map(property(PagePlace.PAGE_ID, PAGE_PLACE.PAGE_ID))
                    .map(property(PagePlace.PLACE_ID, PAGE_PLACE.PLACE_ID))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public PagePlaceRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить все записи таблицы {@code page_place}
     */
    public List<PagePlace> getAll() {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(PAGE_PLACE)
                .fetch(MAPPER::fromDb);
    }

    public void add(Collection<PagePlace> addRecords) {
        new InsertHelper<>(dslContextProvider.ppcdict(), PAGE_PLACE)
                .addAll(MAPPER, addRecords)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    public void delete(Collection<PagePlace> deleteRecords) {
        if (deleteRecords.isEmpty()) {
            return;
        }

        DeleteConditionStep<PagePlaceRecord> step = dslContextProvider.ppcdict()
                .deleteFrom(PAGE_PLACE)
                .where(DSL.falseCondition());
        deleteRecords.forEach(pp -> step.or(PAGE_PLACE.PAGE_ID.eq(pp.getPageId())
                .and(PAGE_PLACE.PLACE_ID.eq(pp.getPlaceId()))));

        step.execute();
    }
}
