package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.PagePlace;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables.PAGESLOTSEQUENCE;
import static ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtPageSlotSequence.PAGEID;
import static ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtPageSlotSequence.PLACEID;
import static ru.yandex.direct.utils.StringUtils.joinLongsToString;

@Repository
@ParametersAreNonnullByDefault
public class PagePlaceYtRepository {
    private final YtProvider ytProvider;
    private final InternalYaBsClusterChooser clusterChooser;

    public PagePlaceYtRepository(YtProvider ytProvider, InternalYaBsClusterChooser clusterChooser) {
        this.ytProvider = ytProvider;
        this.clusterChooser = clusterChooser;
    }

    /**
     * Получить все записи для пейджей внутренней рекламы из yt-таблицы
     */
    public List<PagePlace> getPagePlaces(Collection<Long> pageIds) {
        var whereClause = String.format("%s in (%s)", PAGEID.getName(), joinLongsToString(pageIds));
        var query = String.format("%s, %s FROM [%s] WHERE %s",
                PAGEID.getName(),
                PLACEID.getName(),
                PAGESLOTSEQUENCE.getPath(),
                whereClause);
        List<YtCluster> clustersByPriority = clusterChooser.getAvailableClustersOrdered();
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getDynamicOperator,
                operator -> {
                    var rowset = operator.selectRows(query);
                    return rowset.getYTreeRows().stream()
                            .map(PagePlaceYtRepository::convertFromYt)
                            .collect(toList());
                });
    }

    private static PagePlace convertFromYt(YTreeMapNode node) {
        return new PagePlace()
                .withPageId(node.getLong(PAGEID.getName()))
                .withPlaceId(node.getLong(PLACEID.getName()));
    }

    /**
     * Получить время последнего обновления данных для таблицы {@link YtDbTables#PAGESLOTSEQUENCE}
     * Время последнего обновления хранится в атрибуте таблицы {@link InternalYaBsClusterChooser#MAX_UNIX_TIME}
     * Считаем, что атрибут проставлен и имеет корректное значение, иначе упадем при определении свежести кластера {@link InternalYaBsClusterChooser}
     */
    @Nonnull
    public Long getLastUpdateUnixTime() {
        List<YtCluster> clustersByPriority = clusterChooser.getAvailableClustersOrdered();
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getOperator,
                operator -> operator.readTableNumericAttribute(PAGESLOTSEQUENCE, InternalYaBsClusterChooser.MAX_UNIX_TIME));
    }
}
