package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.InternalAdPlace;
import ru.yandex.direct.dbschema.ppcdict.tables.records.InternalAdPlacesRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.InternalAdPlaces.INTERNAL_AD_PLACES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@ParametersAreNonnullByDefault
@Repository
public class PlaceRepository {
    private static final JooqMapperWithSupplier<InternalAdPlace> MAPPER =
            JooqMapperWithSupplierBuilder.builder(InternalAdPlace::new)
                    .map(property(InternalAdPlace.ID, INTERNAL_AD_PLACES.PLACE_ID))
                    .map(property(InternalAdPlace.PARENT_ID, INTERNAL_AD_PLACES.PARENT_PLACE_ID))
                    .map(property(InternalAdPlace.DESCRIPTION, INTERNAL_AD_PLACES.DESCRIPTION))
                    .build();

    private final DslContextProvider dslContextProvider;

    public PlaceRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить все записи таблицы {@code internal_ad_places}
     *
     * @return список площадок
     */
    public List<InternalAdPlace> getAll() {
        return get(DSL.trueCondition());
    }

    /**
     * Получить записи по placeId из таблицы {@code internal_ad_places}
     *
     * @param placeIds — ID площадок
     * @return список площадок
     */
    public List<InternalAdPlace> getByPlaceIds(Collection<Long> placeIds) {
        return get(INTERNAL_AD_PLACES.PLACE_ID.in(placeIds));
    }

    /**
     * Получить записи, удовлетворяющие переданному условию, из таблицы {@code internal_ad_places}
     *
     * @param condition - условие
     * @return список площадок
     */
    private List<InternalAdPlace> get(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(INTERNAL_AD_PLACES)
                .where(condition)
                .fetch(MAPPER::fromDb);
    }

    public void addOrUpdate(Collection<InternalAdPlace> records) {
        InsertHelper<InternalAdPlacesRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), INTERNAL_AD_PLACES).addAll(MAPPER, records);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(INTERNAL_AD_PLACES.PLACE_ID, MySQLDSL.values(INTERNAL_AD_PLACES.PLACE_ID))
                    .set(INTERNAL_AD_PLACES.PARENT_PLACE_ID, MySQLDSL.values(INTERNAL_AD_PLACES.PARENT_PLACE_ID))
                    .set(INTERNAL_AD_PLACES.DESCRIPTION, MySQLDSL.values(INTERNAL_AD_PLACES.DESCRIPTION));
        }

        insertHelper.executeIfRecordsAdded();
    }

    public void delete(Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }

        dslContextProvider.ppcdict()
                .deleteFrom(INTERNAL_AD_PLACES)
                .where(INTERNAL_AD_PLACES.PLACE_ID.in(ids))
                .execute();
    }
}
