package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.DeleteConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.dbschema.ppcdict.tables.records.TemplatePlaceRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.TemplatePlace.TEMPLATE_PLACE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class TemplatePlaceRepository {

    private static final JooqMapperWithSupplier<TemplatePlace> MAPPER =
            JooqMapperWithSupplierBuilder.builder(TemplatePlace::new)
                    .map(property(TemplatePlace.TEMPLATE_ID, TEMPLATE_PLACE.TEMPLATE_ID))
                    .map(property(TemplatePlace.PLACE_ID, TEMPLATE_PLACE.PLACE_ID))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public TemplatePlaceRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить все записи таблицы {@code template_place}
     */
    public List<TemplatePlace> getAll() {
        return get(DSL.trueCondition());
    }

    /**
     * Получить записи по placeId из таблицы {@code template_place}
     */
    public List<TemplatePlace> getByPlaceIds(Collection<Long> placeIds) {
        return get(TEMPLATE_PLACE.PLACE_ID.in(placeIds));
    }

    /**
     * Получить записи по templateId из таблицы {@code template_place}
     * @param templateId — ID шаблона
     * @return список связок площадки и шаблона
     */
    public List<TemplatePlace> getByTemplateId(Long templateId) {
        return get(TEMPLATE_PLACE.TEMPLATE_ID.eq(templateId));
    }

    /**
     * Для каких плейсов есть шаблоны?
     * Возвращает уникальные ID
     */
    public List<Long> getPlaces() {
        return dslContextProvider.ppcdict()
                .selectDistinct(TEMPLATE_PLACE.PLACE_ID)
                .from(TEMPLATE_PLACE)
                .fetch(TEMPLATE_PLACE.PLACE_ID);
    }

    /**
     * Получить записи удовлетворяющие переданному условию из таблицы {@code template_place}
     */
    private List<TemplatePlace> get(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(TEMPLATE_PLACE)
                .where(condition)
                .fetch(MAPPER::fromDb);
    }

    public void add(Collection<TemplatePlace> addRecords) {
        new InsertHelper<>(dslContextProvider.ppcdict(), TEMPLATE_PLACE)
                .addAll(MAPPER, addRecords)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    public void delete(Collection<TemplatePlace> deleteRecords) {
        if (deleteRecords.isEmpty()) {
            return;
        }

        DeleteConditionStep<TemplatePlaceRecord> step = dslContextProvider.ppcdict()
                .deleteFrom(TEMPLATE_PLACE)
                .where(DSL.falseCondition());
        deleteRecords.forEach(tp -> step.or(TEMPLATE_PLACE.TEMPLATE_ID.eq(tp.getTemplateId())
                .and(TEMPLATE_PLACE.PLACE_ID.in(tp.getPlaceId()))));

        step.execute();
    }

}
