package ru.yandex.direct.core.entity.internalads.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.TemplateResource;
import ru.yandex.direct.core.entity.internalads.model.TemplateResourceOption;
import ru.yandex.direct.dbschema.ppcdict.tables.records.TemplateResourceRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppcdict.tables.TemplateResource.TEMPLATE_RESOURCE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class TemplateResourceRepository {

    private static final JooqMapperWithSupplier<TemplateResource> MAPPER =
            JooqMapperWithSupplierBuilder.builder(TemplateResource::new)
                    .map(property(TemplateResource.ID, TEMPLATE_RESOURCE.ID))
                    .map(property(TemplateResource.TEMPLATE_ID, TEMPLATE_RESOURCE.TEMPLATE_ID))
                    .map(property(TemplateResource.TEMPLATE_COUNTER_TYPE, TEMPLATE_RESOURCE.TEMPLATE_COUNTER_TYPE))
                    .map(property(TemplateResource.TEMPLATE_PART_NO, TEMPLATE_RESOURCE.TEMPLATE_PART_NO))
                    .map(property(TemplateResource.DESCRIPTION, TEMPLATE_RESOURCE.DESCRIPTION))
                    .map(property(TemplateResource.RESOURCE_TYPE, TEMPLATE_RESOURCE.RESOURCE_TYPE))
                    .map(property(TemplateResource.RESOURCE_NO, TEMPLATE_RESOURCE.RESOURCE_NO))
                    .map(property(TemplateResource.POSITION, TEMPLATE_RESOURCE.POSITION))
                    .map(convertibleEnumSet(TemplateResource.OPTIONS, TEMPLATE_RESOURCE.OPTIONS,
                            TemplateResourceOption::fromTypedValue,
                            TemplateResourceOption::getTypedValue))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public TemplateResourceRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить все записи таблицы {@code template_resource}
     */
    public List<TemplateResource> getAll() {
        return get(DSL.trueCondition());
    }

    /**
     * Получить записи по id из таблицы {@code template_resource}
     */
    public List<TemplateResource> getByIds(Collection<Long> ids) {
        return get(TEMPLATE_RESOURCE.ID.in(ids));
    }

    /**
     * Получить записи по templateId из таблицы {@code template_resource}
     */
    public List<TemplateResource> getByTemplateIds(Collection<Long> templateIds) {
        return get(TEMPLATE_RESOURCE.TEMPLATE_ID.in(templateIds));
    }

    /**
     * Получить записи удовлетворяющие переданному условию из таблицы {@code template_resource}
     */
    private List<TemplateResource> get(Condition condition) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(TEMPLATE_RESOURCE)
                .where(condition)
                .fetch(MAPPER::fromDb);
    }

    public void addOrUpdate(Collection<TemplateResource> addOrUpdateRecords) {
        InsertHelper<TemplateResourceRecord> insertHelper = new InsertHelper<>(dslContextProvider.ppcdict(), TEMPLATE_RESOURCE)
                .addAll(MAPPER, addOrUpdateRecords);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(TEMPLATE_RESOURCE.ID, MySQLDSL.values(TEMPLATE_RESOURCE.ID))
                    .set(TEMPLATE_RESOURCE.TEMPLATE_ID, MySQLDSL.values(TEMPLATE_RESOURCE.TEMPLATE_ID))
                    .set(TEMPLATE_RESOURCE.TEMPLATE_COUNTER_TYPE, MySQLDSL.values(TEMPLATE_RESOURCE.TEMPLATE_COUNTER_TYPE))
                    .set(TEMPLATE_RESOURCE.TEMPLATE_PART_NO, MySQLDSL.values(TEMPLATE_RESOURCE.TEMPLATE_PART_NO))
                    .set(TEMPLATE_RESOURCE.DESCRIPTION, MySQLDSL.values(TEMPLATE_RESOURCE.DESCRIPTION))
                    .set(TEMPLATE_RESOURCE.RESOURCE_TYPE, MySQLDSL.values(TEMPLATE_RESOURCE.RESOURCE_TYPE))
                    .set(TEMPLATE_RESOURCE.RESOURCE_NO, MySQLDSL.values(TEMPLATE_RESOURCE.RESOURCE_NO))
                    .set(TEMPLATE_RESOURCE.POSITION, MySQLDSL.values(TEMPLATE_RESOURCE.POSITION))
                    .set(TEMPLATE_RESOURCE.OPTIONS, MySQLDSL.values(TEMPLATE_RESOURCE.OPTIONS));
        }

        insertHelper.executeIfRecordsAdded();
    }

    public void delete(Collection<Long> templateResourceIds) {
        if (templateResourceIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppcdict()
                .deleteFrom(TEMPLATE_RESOURCE)
                .where(TEMPLATE_RESOURCE.ID.in(templateResourceIds))
                .execute();
    }

}
