package ru.yandex.direct.core.entity.internalads.repository;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.internalads.model.TemplateResource;
import ru.yandex.direct.core.entity.internalads.model.TemplateResourceOption;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtTemplateResourceRow;
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables.TEMPLATERESOURCE;

@Repository
@ParametersAreNonnullByDefault
public class TemplateResourceYtRepository {

    /**
     * Мапа, в котором каждому из расчитываемых опций ресурса ставится в соответствие геттер из модели {@link YtTemplateResourceRow},
     * по значению которого добавляем опцию в набор доступных
     *
     * @see #getOptionGettersMap
     */
    private static final Map<TemplateResourceOption, Function<YtTemplateResourceRow, Boolean>> OPTION_GETTERS_MAP = getOptionGettersMap();

    private final YtProvider ytProvider;
    private final InternalYaBsClusterChooser clusterChooser;

    public TemplateResourceYtRepository(YtProvider ytProvider, InternalYaBsClusterChooser clusterChooser) {
        this.ytProvider = ytProvider;
        this.clusterChooser = clusterChooser;
    }

    /**
     * Получить все записи таблицы TemplateResource из yt-таблицы, экспортируемой из БК
     */
    public List<TemplateResource> getAll() {
        List<YtCluster> clustersByPriority = clusterChooser.getAvailableClustersOrdered();
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getOperator,
                operator -> operator.readTableAndMap(TEMPLATERESOURCE, new YtTemplateResourceRow(), TemplateResourceYtRepository::convertFromYt));
    }

    /**
     * Получить время последнего обновления данных для таблицы {@link YtDbTables#TEMPLATERESOURCE}
     * Время последнего обновления хранится в атрибуте таблицы {@link InternalYaBsClusterChooser#MAX_UNIX_TIME}
     * Считаем, что атрибут проставлен и имеет корректное значение, иначе упадем при определении свежести кластера {@link InternalYaBsClusterChooser}
     */
    @Nonnull
    public Long getLastUpdateUnixTime() {
        List<YtCluster> clustersByPriority = clusterChooser.getAvailableClustersOrdered();
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getOperator,
                operator -> operator.readTableNumericAttribute(TEMPLATERESOURCE, InternalYaBsClusterChooser.MAX_UNIX_TIME));
    }

    private static TemplateResource convertFromYt(YtTemplateResourceRow row) {
        return new TemplateResource()
                .withId(row.getTemplateResourceID())
                .withTemplateId(row.getTemplateID())
                .withTemplateCounterType(row.getTemplateCounterType())
                .withTemplatePartNo(row.getTemplatePartNo())
                .withDescription(row.getTemplateResourceDescription())
                .withResourceType(row.getTemplateResourceType())
                .withResourceNo(row.getTemplateResourceNo())
                .withPosition(row.getPosition())
                .withOptions(convertToSetOptions(row));
    }

    private static Set<TemplateResourceOption> convertToSetOptions(YtTemplateResourceRow row) {
        return EntryStream.of(OPTION_GETTERS_MAP)
                .mapValues(getter -> getter.apply(row))
                .filterValues(BooleanUtils::isTrue)
                .keys()
                .toImmutableSet();
    }

    private static ImmutableMap<TemplateResourceOption, Function<YtTemplateResourceRow, Boolean>> getOptionGettersMap() {
        return ImmutableMap.<TemplateResourceOption, Function<YtTemplateResourceRow, Boolean>>builder()
                .put(TemplateResourceOption.REQUIRED, YtTemplateResourceRow::getOptionsRequired)
                .put(TemplateResourceOption.BANANA_URL, YtTemplateResourceRow::getOptionsBananaUrl)
                .put(TemplateResourceOption.BANANA_IMAGE, YtTemplateResourceRow::getOptionsBananaImage)
                .build();
    }

}
