package ru.yandex.direct.core.entity.internalads.restriction;

import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.EnumUtils;

import ru.yandex.direct.core.entity.image.converter.BannerImageConverter;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta;
import ru.yandex.direct.core.entity.image.model.ImageMdsMetaInfo;
import ru.yandex.direct.validation.result.Defect;

@ParametersAreNonnullByDefault
public abstract class AbstractRestrictionImage implements RestrictionImage {

    private final boolean isStrict;
    private final Set<RestrictionImageFormat> formatsForCheck;

    public AbstractRestrictionImage(boolean isStrict, @Nullable Set<RestrictionImageFormat> formatsForCheck) {
        this.isStrict = isStrict;
        this.formatsForCheck = formatsForCheck;
    }

    @Override
    public final boolean isStrict() {
        return isStrict;
    }

    /**
     * Форматы для которых надо применить ограничение
     * по дефолту(null) означает, что надо применять для всех форматов
     */
    @Nullable
    public Set<RestrictionImageFormat> getFormatsForCheck() {
        return formatsForCheck;
    }

    /**
     * Метод для проверок в конечных Restriction
     * Общие проверки делаются в {@link #check(BannerImageFormat)}
     */
    @Nullable
    abstract Defect checkRestriction(BannerImageFormat bannerImageFormat);

    @Nullable
    @Override
    public final Defect check(@Nullable BannerImageFormat bannerImageFormat) {
        if (bannerImageFormat == null) {
            return null;
        }

        Optional<RestrictionImageFormat> origFormat = getImageOrigFormat(bannerImageFormat);
        if (formatsForCheck != null && (origFormat.isEmpty() || !formatsForCheck.contains(origFormat.get()))) {
            return null;
        }

        return checkRestriction(bannerImageFormat);
    }

    static Optional<RestrictionImageFormat> getImageOrigFormat(BannerImageFormat bannerImageFormat) {
        ImageMdsMeta imageMdsMeta = BannerImageConverter.toImageMdsMeta(bannerImageFormat.getMdsMeta());
        return Optional.ofNullable(imageMdsMeta)
                .map(ImageMdsMeta::getMeta)
                .map(ImageMdsMetaInfo::getOrigFormat)
                .map(stringFormat -> EnumUtils.getEnum(RestrictionImageFormat.class, stringFormat));
    }

}
