package ru.yandex.direct.core.entity.internalads.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.utils.UrlUtils;
import ru.yandex.direct.utils.model.UrlParts;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class InternalAdUrlMacrosService {
    public static final String APP_METRICA_PATTERN =
            "(http(s)?:)?\\/\\/(\\w.*)?(redirect.)?appmetri(c|k)a.yandex.(ru|com)\\/*";
    public static final String APPS_FLYER_PATTERN = "(http(s)?:)?\\/\\/app.appsflyer.com";
    public static final String ADJUST_PATTERN = "(http(s)?:)?\\/\\/app.adjust.com";

    private static final Map<InternalAdTrackerType, InternalAdUrlSpecificHandler> TRACKER_HANDLERS;

    static {
        TRACKER_HANDLERS = ImmutableMap.<InternalAdTrackerType,
                InternalAdUrlSpecificHandler>builder()

                .put(InternalAdTrackerType.APPSFLYER,
                        new InternalAdUrlSpecificHandlerBuilder(APPS_FLYER_PATTERN)
                                .defaultParameter("clickid", "{LOGID}")
                                .defaultParameter("google_aid", "{GOOGLEAID}")
                                .defaultParameter("android_id", "{ANDROIDID}")
                                .defaultParameter("ios_ifa", "{IOSIFA}")
                                .build())

                .put(InternalAdTrackerType.ADJUST,
                        new InternalAdUrlSpecificHandlerBuilder(ADJUST_PATTERN)
                                .defaultParameter("google_aid_lc_sh1", "{GOOGLE_AID_LC_SH1}")
                                .defaultParameter("android_id_lc_sh1", "{ANDROID_ID_LC_SH1}")
                                .defaultParameter("idfa", "{idfa}")
                                .defaultParameter("ua", "{user_agent}")
                                .defaultParameter("package_name", "{app_name}")
                                .defaultParameter("ip", "{ip_address}")
                                .defaultParameter("adjust-adid", "{adid}")
                                .defaultParameter("android-id-md5", "{android_id_md5}")
                                .defaultParameter("mac-address-sha1", "{mac_sha1}")
                                .defaultParameter("mac-address-md5", "{mac_md5}")
                                .defaultParameter("idfa-android-id", "{idfa||android_id}")
                                .defaultParameter("idfa-gps-adid", "{idfa||gps_adid}")
                                .defaultParameter("ios-idfa-base64md5", "{idfa_md5}")
                                .defaultParameter("ios-idfa-hexmd5", "{idfa_md5_hex}")
                                .defaultParameter("idfv", "{idfv}")
                                .defaultParameter("user-activity-type", "{activity_kind}")
                                .defaultParameter("click-timestamp", "{click_time}")
                                .defaultParameter("conversion-timestamp", "{installed_at}")
                                .defaultParameter("connection-type", "{connection_type}")
                                .defaultParameter("isp", "{isp}")
                                .defaultParameter("region-code", "{region}")
                                .defaultParameter("country-code", "{country}")
                                .defaultParameter("country-subdivision", "{country_subdivision}")
                                .defaultParameter("city", "{city}")
                                .defaultParameter("postal-code", "{postal_code}")
                                .defaultParameter("device-model", "{device_name}")
                                .defaultParameter("device-type", "{device_type}")
                                .defaultParameter("os", "{os_name}")
                                .defaultParameter("api-level", "{api_level}")
                                .defaultParameter("device-sdk-version", "{sdk_version}")
                                .defaultParameter("os-version", "{os_version}")
                                .defaultParameter("environment", "{environment}")
                                .defaultParameter("device-timezone", "{timezone}")
                                .defaultParameter("install_callback",
                                        "http://postback.yandexadexchange.net/postback?reqid={LOGID}")
                                .build())

                .put(InternalAdTrackerType.APPMETRICA,
                        new InternalAdUrlSpecificHandlerBuilder(APP_METRICA_PATTERN)
                                .defaultParameter("click_id", "{LOGID}")
                                .defaultParameter("google_aid", "{GOOGLE_AID_LC}")
                                .defaultParameter("ios_ifa", "{IDFA_UC}")
                                .build())

                .build();

        checkState(Arrays.stream(InternalAdTrackerType.values()).allMatch(TRACKER_HANDLERS::containsKey));
    }

    /**
     * Проставляет для каждого баннера с resourceType = URL дефолтные макросы в зависимости от домена
     */
    @Nullable
    public static List<TemplateVariable> enrichDefaultUrlParametrsForBanner(BannerWithInternalInfo banner,
                                                                            Map<Long, InternalTemplateInfo> templateIdToTemplateInfo) {
        var internalTemplateInfo = templateIdToTemplateInfo.get(banner.getTemplateId());
        var resourceIdToType = listToMap(internalTemplateInfo.getResources(), ResourceInfo::getId,
                ResourceInfo::getType);

        banner.setTemplateVariables(mapList(banner.getTemplateVariables(),
                templateVariable -> addDefaultUrlParametrs(templateVariable, resourceIdToType)));

        return banner.getTemplateVariables();
    }

    private static TemplateVariable addDefaultUrlParametrs(TemplateVariable templateVariable,
                                                           Map<Long, ResourceType> resourceIdToType) {
        var resourceType = resourceIdToType.get(templateVariable.getTemplateResourceId());
        var url = templateVariable.getInternalValue();
        if (resourceType != ResourceType.URL || url == null) {
            return templateVariable;
        }
        return templateVariable.withInternalValue(normalizeTrackingUrl(url));
    }

    public static String normalizeTrackingUrl(String trackingUrl) {
        UrlParts urlParts = UrlUtils.laxParseUrl(trackingUrl);
        var updatedParameters = addDefaultTrackingParameters(trackingUrl, nvl(urlParts.getParameters(),
                emptyList()));

        UrlParts resultingUrl = urlParts.toBuilder()
                .withProtocol(urlParts.getProtocol())
                .withPath(urlParts.getPath().isEmpty() ? "/" : urlParts.getPath())
                .withParameters(updatedParameters)
                .build();

        return resultingUrl.toUrl();
    }

    private static List<Pair<String, String>> addDefaultTrackingParameters(String url,
                                                                           List<Pair<String, String>> queryParameters) {

        Optional<InternalAdUrlSpecificHandler> trackerHandlerOptional =
                Arrays.stream(InternalAdTrackerType.values())
                        .map(TRACKER_HANDLERS::get)
                        .filter(handler -> handler.handlesDomain(url))
                        .findFirst();

        return trackerHandlerOptional
                .map(handler -> mergeDefaultAndCurrentParameters(handler, queryParameters))
                .orElse(queryParameters);
    }

    private static List<Pair<String, String>> mergeDefaultAndCurrentParameters(InternalAdUrlSpecificHandler handler,
                                                                               List<Pair<String, String>> parameters) {
        Map<String, Pair<String, String>> defaultParametersByName = handler.getDefaultParametersByName();

        List<Pair<String, String>> result = StreamEx.of(parameters)
                .map(parameter -> {
                    String name = parameter.getKey();
                    if (defaultParametersByName.containsKey(name)) {
                        return defaultParametersByName.get(name);
                    }
                    return parameter;
                })
                .toCollection(ArrayList::new);
        Set<String> seenParameterNames = listToSet(parameters, Pair::getKey);


        Collection<Pair<String, String>> defaultParameters = handler.getDefaultParameters();
        result.addAll(StreamEx.of(defaultParameters)
                .filter(definedParameter -> !seenParameterNames.contains(definedParameter.getKey()))
                .toList());

        return result;
    }

}
