package ru.yandex.direct.core.entity.internalads.service;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.model.InternalAdsManagerProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsManagerProductAccessType;
import ru.yandex.direct.core.entity.internalads.repository.InternalAdsManagerProductAccessRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacClientsRelationsStorage;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.rbac.model.ClientsRelationType.INTERNAL_AD_PUBLISHER;
import static ru.yandex.direct.rbac.model.ClientsRelationType.INTERNAL_AD_READER;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Доступ менеджеров внутренней рекламы к продуктам внутренней рекламы
 * <p>
 * Сервис манипулирует объектами {@link InternalAdsManagerProductAccess}, которые
 * инкапсулируют в себе тип доступа и список плейсов, если доступ частичный;
 * работу с данными в clients_relations берёт на себя (и делегирует её
 * RbacClientsRelations)
 * <p>
 * Все операции записи выполняются в транзакциях
 */
@ParametersAreNonnullByDefault
@Service
public class InternalAdsManagerProductAccessService {
    private static final Set<ClientsRelationType> INTERNAL_AD_RELATION_TYPES =
            Collections.unmodifiableSet(EnumSet.of(INTERNAL_AD_PUBLISHER, INTERNAL_AD_READER));

    private final RbacService rbacService;
    private final UserService userService;
    private final RbacClientsRelations rbacClientsRelations;
    private final RbacClientsRelationsStorage rbacClientsRelationsStorage;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final PlaceService placeService;
    private final InternalAdsManagerProductAccessRepository internalAdsManagerProductAccessRepository;

    @Autowired
    public InternalAdsManagerProductAccessService(
            RbacService rbacService,
            UserService userService,
            RbacClientsRelations rbacClientsRelations,
            RbacClientsRelationsStorage rbacClientsRelationsStorage,
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider,
            PlaceService placeService,
            InternalAdsManagerProductAccessRepository internalAdsManagerProductAccessRepository) {
        this.rbacService = rbacService;
        this.userService = userService;
        this.rbacClientsRelations = rbacClientsRelations;
        this.rbacClientsRelationsStorage = rbacClientsRelationsStorage;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.placeService = placeService;
        this.internalAdsManagerProductAccessRepository = internalAdsManagerProductAccessRepository;
    }

    /**
     * для нескольких маркетологов получает информацию о доступе ко всем продуктам,
     * куда у них этот доступ есть
     *
     * @param managerClientIds {@link ClientId} менеджеров
     * @return {@link Map}, {@link ClientId} менеджера в качестве ключа и список доступов в качестве значения
     */
    public Map<ClientId, List<InternalAdsManagerProductAccess>> getProductAccessForManagers(
            Collection<ClientId> managerClientIds) {
        Map<Long, List<ClientsRelation>> managerToRelations =
                rbacClientsRelations.getRelatedProductsForInternalAdManagers(managerClientIds);

        List<ClientsRelation> allPublisherRelations = managerToRelations.values().stream()
                .flatMap(List::stream)
                .filter(relation -> relation.getRelationType() == INTERNAL_AD_PUBLISHER)
                .collect(Collectors.toList());

        ShardedData<ClientsRelation> clientsRelationShardedData = shardHelper.groupByShard(allPublisherRelations,
                ShardKey.CLIENT_ID,
                ClientsRelation::getClientIdTo);

        Map<Long, List<Long>> relationIdToPlaceIds = clientsRelationShardedData.stream()
                .flatMapKeyValue(this::getRelationPlaceIdsOnShard)
                .toMap(Map.Entry::getKey, Map.Entry::getValue);

        return EntryStream.of(managerToRelations)
                .mapKeys(ClientId::fromLong)
                .mapValues(relations -> mapList(relations, relation -> getInternalAdsManagerProductAccess(relation,
                        relationIdToPlaceIds.getOrDefault(relation.getRelationId(), List.of()))))
                .toMap();
    }

    /**
     * возвращает все доступы одного маркетолога. если доступов нет, возвращает пустой список.
     */
    public List<InternalAdsManagerProductAccess> getProductAccessForSingleManager(ClientId managerClientId) {
        return getProductAccessForManagers(List.of(managerClientId)).getOrDefault(managerClientId, emptyList());
    }

    private Stream<? extends Map.Entry<Long, List<Long>>> getRelationPlaceIdsOnShard(
            int shard,
            List<ClientsRelation> relationsOnShard) {
        return EntryStream.of(internalAdsManagerProductAccessRepository.getPlaceIdsForMultipleRelations(shard,
                mapList(relationsOnShard, ClientsRelation::getRelationId)));
    }

    @Nullable
    public InternalAdsManagerProductAccess getManagerAccessToSingleProduct(
            ClientId managerClientId, ClientId productClientId) {
        Optional<ClientsRelation> relationOptional =
                rbacClientsRelations.getInternalAdProductRelation(managerClientId, productClientId);

        if (relationOptional.isEmpty()) {
            return null;
        }

        ClientsRelation relation = relationOptional.get();

        Set<Long> placeIds = internalAdsManagerProductAccessRepository.getAccessiblePlaceIds(
                shardHelper.getShardByClientIdStrictly(productClientId), relation.getRelationId());

        return getInternalAdsManagerProductAccess(relation, placeIds);
    }

    public List<InternalAdsManagerProductAccess> getManagerAccessToMultipleProducts(
            ClientId managerClientId, Collection<ClientId> productClientIds) {
        List<ClientsRelation> relations = rbacClientsRelations.getInternalAdProductRelationsForMultipleProducts(
                managerClientId, productClientIds);

        Map<Long, List<Long>> placeIdsByRelationId = shardHelper.groupByShard(relations, ShardKey.CLIENT_ID,
                ClientsRelation::getClientIdTo).stream()
                .flatMapKeyValue((shard, relationsOnShard) -> {
                    Map<Long, List<Long>> placeIdMapOnShard = internalAdsManagerProductAccessRepository
                            .getPlaceIdsForMultipleRelations(shard,
                                    mapList(relationsOnShard, ClientsRelation::getRelationId));
                    return EntryStream.of(placeIdMapOnShard);
                })
                .toMap(Map.Entry::getKey, Map.Entry::getValue);

        return relations.stream()
                .map(relation -> getInternalAdsManagerProductAccess(relation,
                        placeIdsByRelationId.getOrDefault(relation.getRelationId(), emptyList())))
                .collect(Collectors.toList());
    }

    /**
     * записать в базу новый доступ
     * проверки, что такого доступа ещё нет, не делаются, ожидается, что для этого у потребителя есть валидатор
     *
     * @param productAccess какой доступ
     * @throws RuntimeException если в базе уже есть доступ этого маркетолога к этому продукту
     */
    public void addProductAccess(InternalAdsManagerProductAccess productAccess) {
        ClientId managerClientId = productAccess.getManagerClientId();
        ClientId productClientId = productAccess.getProductClientId();
        ClientsRelationType relationType =
                productAccess.getAccessType() == InternalAdsManagerProductAccessType.READONLY ?
                        INTERNAL_AD_READER :
                        INTERNAL_AD_PUBLISHER;

        int shard = shardHelper.getShardByClientIdStrictly(productClientId);
        dslContextProvider.ppc(shard).transaction(ctx -> {
            DSLContext dslContext = ctx.dsl();

            rbacClientsRelations.addInternalAdProductRelation(
                    dslContext, managerClientId, productClientId, relationType);

            Long relationId = rbacClientsRelationsStorage.getRelationId(
                    dslContext, managerClientId, productClientId, relationType);

            internalAdsManagerProductAccessRepository.savePlaces(dslContext, relationId, productAccess.getPlaceIds());
        });
    }

    /**
     * поменять уже существующий доступ
     * проверок, что этот доступ уже есть в базе, нет, ожидается, что у потребителя для этого есть валидатор,
     * но текущая реализация упадёт, если доступа не было
     *
     * @param productAccess какой доступ надо сделать
     * @throws RuntimeException если доступа не было
     */
    public void updateProductAccess(InternalAdsManagerProductAccess productAccess) {
        ClientId managerClientId = productAccess.getManagerClientId();
        ClientId productClientId = productAccess.getProductClientId();
        ClientsRelationType relationType =
                productAccess.getAccessType() == InternalAdsManagerProductAccessType.READONLY ?
                        INTERNAL_AD_READER :
                        INTERNAL_AD_PUBLISHER;

        int shard = shardHelper.getShardByClientIdStrictly(productClientId);
        dslContextProvider.ppc(shard).transaction(ctx -> {
            DSLContext dslContext = ctx.dsl();

            rbacClientsRelations.updateInternalAdProductRelation(
                    dslContext, managerClientId, productClientId, relationType);

            Long relationId = rbacClientsRelationsStorage.getRelationId(
                    dslContext, managerClientId, productClientId, relationType);
            checkNotNull(relationId, "relationId is null, trying to update a relation that didn't exist?");

            internalAdsManagerProductAccessRepository.cleanUpPlaces(dslContext, List.of(relationId));
            internalAdsManagerProductAccessRepository.savePlaces(dslContext, relationId, productAccess.getPlaceIds());
        });
    }

    /**
     * упраздняет все доступы маркетолога к нескольким продуктам
     * проверок, что эти доступы уже были, не делается; если они нужны, потребителю нужно сделать для этого валидатор
     * текущая реализация, если доступов нет, молча пропускает соответствующие {@link ClientId} продуктов
     *
     * @param managerClientId  {@link ClientId} маркетолога
     * @param productClientIds {@link ClientId} продуктов
     */
    public void removeAccessToMultipleProducts(ClientId managerClientId, Collection<ClientId> productClientIds) {
        ShardedData<ClientId> shardedProducts = shardHelper.groupByShard(
                productClientIds, ShardKey.CLIENT_ID, ClientId::asLong);

        shardedProducts.forEach((shard, productClientIdsOnShard) -> {
            List<Long> relationIds = rbacClientsRelationsStorage.getRelationIds(shard,
                    managerClientId, productClientIds, INTERNAL_AD_RELATION_TYPES);

            dslContextProvider.ppc(shard).transaction(ctx -> {
                DSLContext dslContext = ctx.dsl();
                rbacClientsRelations.removeInternalAdProductsRelations(dslContext, managerClientId, productClientIds);

                internalAdsManagerProductAccessRepository.cleanUpPlaces(dslContext, relationIds);
            });
        });
    }

    private static InternalAdsManagerProductAccess getInternalAdsManagerProductAccess(
            ClientsRelation relation, @Nullable Collection<Long> placeIds) {
        ClientsRelationType relationType = relation.getRelationType();
        checkState(INTERNAL_AD_RELATION_TYPES.contains(relationType), "Unexpected value: " + relationType);

        @Nullable final Set<Long> accessPlaceIds;
        final InternalAdsManagerProductAccessType accessType;
        if (relationType == INTERNAL_AD_READER) {
            accessType = InternalAdsManagerProductAccessType.READONLY;
            accessPlaceIds = null;
        } else { // relationType == INTERNAL_AD_PUBLISHER
            checkNotNull(placeIds, "places must not be null for accessType = INTERNAL_AD_PUBLISHER");

            if (placeIds.isEmpty()) {
                accessType = InternalAdsManagerProductAccessType.FULL;
                accessPlaceIds = null;
            } else {
                accessType = InternalAdsManagerProductAccessType.PARTIAL;
                accessPlaceIds = new HashSet<>(placeIds);
            }
        }

        return new InternalAdsManagerProductAccess(ClientId.fromLong(relation.getClientIdFrom()),
                ClientId.fromLong(relation.getClientIdTo()), accessType, accessPlaceIds);
    }

}
