package ru.yandex.direct.core.entity.internalads.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.model.InternalAdPlaceInfo;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsFullProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsManagerProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsNoProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsOperatorProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsPartialProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsReadAllProductAccess;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class InternalAdsOperatorProductAccessService {
    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final PlaceService placeService;
    private final InternalAdsManagerProductAccessService internalAdsManagerProductAccessService;

    @Autowired
    public InternalAdsOperatorProductAccessService(
            ShardHelper shardHelper, RbacService rbacService, PlaceService placeService,
            InternalAdsManagerProductAccessService internalAdsManagerProductAccessService) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.placeService = placeService;
        this.internalAdsManagerProductAccessService = internalAdsManagerProductAccessService;
    }

    public InternalAdsOperatorProductAccess getAccess(Long operatorUid, ClientId productClientId) {
        List<InternalAdsOperatorProductAccess> accesses =
                getAccessToMultipleProducts(operatorUid, List.of(productClientId));

        if (accesses.isEmpty()) {
            return null;
        }

        return accesses.get(0);
    }

    public List<InternalAdsOperatorProductAccess> getAccessToMultipleProducts(
            Long operatorUid, Collection<ClientId> productClientIds) {
        RbacRole operatorRole = rbacService.getUidRole(operatorUid);
        switch (operatorRole) {
            case SUPER:
            case INTERNAL_AD_ADMIN:
                return mapList(productClientIds,
                        productClientId -> new InternalAdsFullProductAccess(operatorUid, productClientId,
                                () -> listToSet(placeService.getPlaceInfoForValidPlaces(),
                                        InternalAdPlaceInfo::getId)));

            case SUPERREADER:
            case INTERNAL_AD_SUPERREADER:
                return mapList(productClientIds,
                        productClientId -> new InternalAdsReadAllProductAccess(operatorUid, productClientId));

            case INTERNAL_AD_MANAGER:
                Long managerClientId = checkNotNull(shardHelper.getClientIdByUid(operatorUid));

                List<InternalAdsManagerProductAccess> accesses =
                        internalAdsManagerProductAccessService.getManagerAccessToMultipleProducts(
                                ClientId.fromLong(managerClientId), productClientIds);

                Map<ClientId, InternalAdsManagerProductAccess> accessByProductClientId = accesses.stream()
                        .collect(Collectors.toMap(InternalAdsManagerProductAccess::getProductClientId,
                                Function.identity()));

                return productClientIds.stream()
                        .map(productClientId -> {
                            if (!accessByProductClientId.containsKey(productClientId)) {
                                return new InternalAdsNoProductAccess(operatorUid, productClientId);
                            }

                            InternalAdsManagerProductAccess access = accessByProductClientId.get(productClientId);
                            switch (access.getAccessType()) {
                                case FULL:
                                    return new InternalAdsFullProductAccess(operatorUid, productClientId,
                                            () -> listToSet(placeService.getPlaceInfoForValidPlaces(),
                                                    InternalAdPlaceInfo::getId));
                                case READONLY:
                                    return new InternalAdsReadAllProductAccess(operatorUid, productClientId);
                                case PARTIAL:
                                    return new InternalAdsPartialProductAccess(operatorUid, productClientId,
                                            checkNotNull(access.getPlaceIds()));

                                default:
                                    throw new IllegalStateException("Unexpected value: " + access.getAccessType());
                            }
                        })
                        .collect(Collectors.toList());

            default:
                return mapList(productClientIds,
                        productClientId -> new InternalAdsNoProductAccess(operatorUid, productClientId));
        }
    }
}
