package ru.yandex.direct.core.entity.internalads.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ClientWithUsers;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsManagerProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductWithClient;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductWithClientAndAccess;
import ru.yandex.direct.core.entity.internalads.repository.InternalAdsProductRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Управление данными продуктов внутренней рекламы.
 * <p>
 * С точки зрения интерфейсов продукт внутренней рекламы -- это корзинка
 * для кампаний, например, Яндекс.Браузера для мобильных.
 * <p>
 * С точки зрения реализации продукт внутренней рекламы -- это клиент,
 * у которого стоит internal_ad_product в clients.perms и есть запись
 * в internal_ad_products. Этот сервис отвечает за данные в internal_ad_products.
 */
@ParametersAreNonnullByDefault
@Service
public class InternalAdsProductService {
    private final ShardHelper shardHelper;
    private final InternalAdsProductRepository internalAdsProductRepository;
    private final InternalAdsManagerProductAccessService internalAdsManagerProductAccessService;
    private final ClientRepository clientRepository;
    private final RbacService rbacService;

    @Autowired
    public InternalAdsProductService(
            ShardHelper shardHelper,
            InternalAdsProductRepository internalAdsProductRepository,
            InternalAdsManagerProductAccessService internalAdsManagerProductAccessService,
            ClientRepository clientRepository,
            RbacService rbacService) {
        this.shardHelper = shardHelper;
        this.internalAdsProductRepository = internalAdsProductRepository;
        this.internalAdsManagerProductAccessService = internalAdsManagerProductAccessService;
        this.clientRepository = clientRepository;
        this.rbacService = rbacService;
    }

    public boolean createProduct(InternalAdsProduct product) {
        int shard = shardHelper.getShardByClientIdStrictly(product.getClientId());
        return internalAdsProductRepository.createProduct(shard, product) > 0;
    }

    /**
     * получить данные продукта по clientId
     *
     * @throws IllegalArgumentException если нет записи в базе
     */
    public InternalAdsProduct getProduct(ClientId productId) {
        return getProducts(Set.of(productId)).stream().findFirst()
                .orElseThrow(() -> new IllegalStateException("product not found. clientId = " + productId));
    }

    /**
     * Получить инфорамцию о заданных продуктах внутренней рекламы
     *
     * @param productIds clientId продуктов
     * @throws IllegalArgumentException если количество найденных продуктов не соответствует количеству запрошенных.
     */
    public List<InternalAdsProduct> getProducts(Collection<ClientId> productIds) {

        Collection<Long> ids = mapList(productIds, ClientId::asLong);
        List<InternalAdsProduct> result = new ArrayList<>(productIds.size());
        shardHelper.groupByShard(ids, ShardKey.CLIENT_ID)
                .forEach((shard, shardProductIds) -> result.addAll(internalAdsProductRepository.getProducts(shard,
                        shardProductIds)));
        return result;
    }

    /**
     * получить список ClientID продуктов по имени продукта (product_name)
     * если продукта с таким именем нет, возвращает пустой список
     */
    public List<ClientId> getClientIdsOfProductsByName(String productName) {
        return internalAdsProductRepository.getClientIdsOfProductsByName(productName);
    }

    /**
     * Получить информацию обо всех продуктах внутренней рекламы: данные клиента + данные из internal_ad_products
     *
     * @throws IllegalStateException если есть данные в internal_ad_products, для которых нет клиента с таким clientId
     */
    public List<InternalAdsProductWithClient> getAllProducts() {
        return shardHelper.dbShards().stream()
                .flatMap(shard -> {
                    List<InternalAdsProduct> products = internalAdsProductRepository.getAllProducts(shard);
                    Map<Long, ClientWithUsers> clientWithUsersByClientIdMap = getProductClientInfo(shard, products);

                    return makeInternalAdsProductWithClientStream(products, clientWithUsersByClientIdMap);
                })
                .collect(Collectors.toList());
    }

    private Stream<InternalAdsProductWithClient> makeInternalAdsProductWithClientStream(List<InternalAdsProduct> products,
                                                                                        Map<Long, ClientWithUsers> clientWithUsersByClientId) {
        return products.stream()
                .map(product -> {
                    ClientId clientId = product.getClientId();
                    ClientWithUsers client = clientWithUsersByClientId.get(clientId.asLong());
                    checkState(client != null, "no client with clientId = %s", clientId);
                    return InternalAdsProductWithClient.of(product, client);
                });
    }

    private Stream<InternalAdsProductWithClientAndAccess> makeInternalAdsProductWithClientAndAccessStream(
            List<InternalAdsProduct> products,
            Map<Long, ClientWithUsers> clientWithUsersByClientId,
            Map<Long, InternalAdsManagerProductAccess> accessByProductClientId) {
        return products.stream()
                .map(product -> {
                    ClientId clientId = product.getClientId();
                    ClientWithUsers client = clientWithUsersByClientId.get(clientId.asLong());
                    checkState(client != null, "no client with clientId = %s", clientId);
                    return InternalAdsProductWithClientAndAccess.of(product, client,
                            accessByProductClientId.get(clientId.asLong()));
                });
    }

    /**
     * Получить информацию о клиенте и пользователе для заданных продуктов внутренней рекламы
     *
     * @param products - список продуктов внутренней рекламы
     * @return Мар, где ключем является clientId продукта, а значение ClientWithUsers,
     */
    private Map<Long, ClientWithUsers> getProductClientInfo(int shard, List<InternalAdsProduct> products) {
        return listToMap(clientRepository.getClientData(shard, mapList(products, InternalAdsProduct::getClientId)),
                ClientWithUsers::getClientId,
                Function.identity());
    }

    /**
     * Получить список продуктов доступных для заданного маркетолога (clientId)
     *
     * @param clientId - маркетолог внутренней рекламы
     * @return
     */
    public List<InternalAdsProductWithClientAndAccess> getProductsAccessibleForClientId(ClientId clientId) {
        List<InternalAdsManagerProductAccess> accesses =
                internalAdsManagerProductAccessService.getProductAccessForSingleManager(clientId);

        if (accesses.isEmpty()) {
            return Collections.emptyList();
        }

        ShardedData<InternalAdsManagerProductAccess> shardedAccesses =
                shardHelper.groupByShard(accesses, ShardKey.CLIENT_ID, access -> access.getProductClientId().asLong());

        return shardedAccesses.stream()
                .flatMapKeyValue((shard, accessesOnShard) -> {
                    List<Long> relatedProductIds = mapList(accessesOnShard,
                            access -> access.getProductClientId().asLong());

                    checkArgument(relatedProductIds != null,
                            "relatedProductIds is null! clientId = %s, shard = %s",
                            clientId, shard);

                    List<InternalAdsProduct> products = internalAdsProductRepository.getProducts(shard,
                            relatedProductIds);
                    Map<Long, ClientWithUsers> clientMap = getProductClientInfo(shard, products);
                    Map<Long, InternalAdsManagerProductAccess> accessByProductClientId =
                            listToMap(accessesOnShard, access -> access.getProductClientId().asLong());

                    return makeInternalAdsProductWithClientAndAccessStream(
                            products, clientMap, accessByProductClientId);
                })
                .collect(Collectors.toList());
    }

    public void updateProduct(InternalAdsProduct product) {
        int shard = shardHelper.getShardByClientIdStrictly(product.getClientId());
        internalAdsProductRepository.updateProduct(shard, product);
    }

    /**
     * Можно ли в этом клиенте создавать кампании внутренней рекламы
     */
    public boolean clientCanHaveInternalAdCampaigns(ClientId clientId) {
        return rbacService.isInternalAdProduct(clientId);
    }

}
