package ru.yandex.direct.core.entity.internalads.service;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.core.entity.internalads.model.InternalAdPlace;
import ru.yandex.direct.core.entity.internalads.model.InternalAdPlaceInfo;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplatePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.PlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplatePlaceRepository;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.CollectionUtils.flatToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class PlaceService {
    private static final Logger LOG = LoggerFactory.getLogger(PlaceService.class);

    private final PlaceRepository placeRepository;
    private final TemplatePlaceRepository templatePlaceRepository;
    private final DirectTemplatePlaceRepository directTemplatePlaceRepository;
    private final Supplier<PlaceInfoCalculationResult> placeInfoForValidPlacesSupplier;

    public PlaceService(PlaceRepository placeRepository,
            TemplatePlaceRepository templatePlaceRepository,
            DirectTemplatePlaceRepository directTemplatePlaceRepository) {
        this.placeRepository = placeRepository;
        this.templatePlaceRepository = templatePlaceRepository;
        this.directTemplatePlaceRepository = directTemplatePlaceRepository;

        placeInfoForValidPlacesSupplier =
                Suppliers.memoizeWithExpiration(() -> {
                    try {
                        return PlaceInfoCalculationResult.success(calculatePlaceInfoForValidPlaces());
                    } catch (RuntimeException e) {
                        return PlaceInfoCalculationResult.error(e);
                    }
                }, 30, TimeUnit.MINUTES);
    }

    /**
     * возвращает информацию по плейсам: только плейсы, для которых есть шаблоны в template_place,
     * с информацией про родительские плейсы, упорядоченные лексикографически по "полному описанию":
     * описанию (description) всех плейсов, начиная от корня, через разделитель
     * @throws IllegalStateException если данные в базе сломаны настолько, что получить
     *                               все "полные описания" никак нельзя
     */
    public List<InternalAdPlaceInfo> getPlaceInfoForValidPlaces() {
        return placeInfoForValidPlacesSupplier.get().getPlacesOrThrowUnderlyingException();
    }

    public List<Long> getValidPlaceIds() {
        return mapList(getPlaceInfoForValidPlaces(), InternalAdPlaceInfo::getId);
    }

    /**
     * возвращает информацию по плейсам в зависимости от переданных id
     */
    public List<InternalAdPlaceInfo> getPlaceInfoForValidPlaceByIds(@Nullable Collection<Long> placeIds) {
        if (placeIds == null) {
            return getPlaceInfoForValidPlaces();
        }
        return filterList(getPlaceInfoForValidPlaces(), placeInfo -> placeIds.contains(placeInfo.getId()));
    }

    private List<InternalAdPlaceInfo> calculatePlaceInfoForValidPlaces() {
        List<InternalAdPlace> allPlaces = placeRepository.getAll();
        List<Long> validPlaceIds = filterList(flatToSet(List.of(
                    templatePlaceRepository.getPlaces(), directTemplatePlaceRepository.getPlaces()
                )),
                placeId -> !Constants.INVALID_PLACES.contains(placeId));
        return buildPlaceInfos(allPlaces, validPlaceIds);
    }

    static List<InternalAdPlaceInfo> buildPlaceInfos(List<InternalAdPlace> allPlaces,
                                                     List<Long> placeIds) {
        try (TraceProfile ignore = Trace.current().profile("buildPlaceInfos")) {
            Map<Long, InternalAdPlace> placeById = listToMap(allPlaces, InternalAdPlace::getId);
            return placeIds.stream()
                    .filter(existsModelForPlaceId(placeById))
                    .map(placeById::get)
                    .map(place -> modelToPlaceInfo(place, placeById))
                    .sorted(comparing(InternalAdPlaceInfo::getFullDescription)
                            .thenComparing(InternalAdPlaceInfo::getId))
                    .collect(toList());
        }
    }

    static InternalAdPlaceInfo modelToPlaceInfo(InternalAdPlace place,
                                                Map<Long, InternalAdPlace> placeById) {
        List<String> descriptions = Stream.iterate(place.getId(), placePresent(placeById), getParentPlaceId(placeById))
                .takeWhile(stopAtCycle())
                .map(placeById::get)
                .map(pathPlace -> Objects.requireNonNullElse(
                        pathPlace.getDescription(),
                        pathPlace.getId().toString()))
                .collect(toList());

        String fullDescription = StreamEx.ofReversed(descriptions).joining(" / ");
        return new InternalAdPlaceInfo(place.getId(), fullDescription);
    }

    private static UnaryOperator<Long> getParentPlaceId(Map<Long, InternalAdPlace> placeById) {
        return placeId -> {
            Long parentId = placeById.get(placeId).getParentId();

            if (parentId != null && parentId != 0L && !placeById.containsKey(parentId)) {
                LOG.warn("failed to find place record for parent placeId = {}, referenced from placeId = {}",
                        parentId, placeId);
            }

            return parentId;
        };
    }

    private static Predicate<Long> placePresent(Map<Long, InternalAdPlace> placeById) {
        return placeId -> placeId != null && placeId != 0L && placeById.containsKey(placeId);
    }

    private static Predicate<Long> existsModelForPlaceId(Map<Long, InternalAdPlace> placeById) {
        return placeId -> {
            if (!placeById.containsKey(placeId)) {
                LOG.warn("failed to find place record for place = {}", placeId);
                return false;
            }
            return true;
        };
    }

    private static Predicate<Long> stopAtCycle() {
        Set<Long> seenPlaceId = new LinkedHashSet<>();
        return placeId -> {
            if (seenPlaceId.contains(placeId)) {
                LOG.warn("place graph is not cycle-free, found a cycle: {}}", seenPlaceId);
                return false;
            }

            seenPlaceId.add(placeId);

            return true;
        };
    }

    private static class PlaceInfoCalculationResult {
        @Nullable
        final List<InternalAdPlaceInfo> places;
        @Nullable
        final RuntimeException calculationException;

        PlaceInfoCalculationResult(@Nullable List<InternalAdPlaceInfo> places,
                                   @Nullable RuntimeException calculationException) {
            checkState(places == null ^ calculationException == null, "must have either a result or an exception");
            this.places = places;
            this.calculationException = calculationException;
        }

        static PlaceInfoCalculationResult success(List<InternalAdPlaceInfo> places) {
            return new PlaceInfoCalculationResult(checkNotNull(places), null);
        }

        static PlaceInfoCalculationResult error(RuntimeException error) {
            return new PlaceInfoCalculationResult(null, checkNotNull(error));
        }

        List<InternalAdPlaceInfo> getPlacesOrThrowUnderlyingException() {
            if (places != null) {
                return places;
            }

            throw new IllegalStateException("failed to get places", calculationException);
        }
    }
}
