package ru.yandex.direct.core.entity.internalads.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.internalads.model.InternalTemplateOverrides;
import ru.yandex.direct.core.entity.internalads.restriction.Restriction;
import ru.yandex.direct.core.entity.internalads.restriction.Restrictions;

import static ru.yandex.direct.core.entity.internalads.Constants.BANNER_ON_404_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.Constants.DESKTOP_BANNER_ON_MORDA_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.Constants.MEDIA_BANNER_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.Constants.TEASER_INLINE_BROWSER_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.Constants.TEASER_INLINE_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.Constants.TEASER_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.Constants.TEASER_WITH_TIME_TEMPLATE_ID;
import static ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageFormat.GIF;
import static ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageFormat.JPEG;
import static ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageFormat.PNG;
import static ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageFormat.SVG;
import static ru.yandex.direct.core.entity.internalads.restriction.Restrictions.DEFAULT_IMAGE_MAX_SIZE_KB;

/**
 * Дополнительная информация о шаблоне и его ресурсах.
 * Переопределяет и дополняет информацию из {@link TemplateResourceService}.
 * <p>
 * https://wiki.yandex-team.ru/users/polevikova/banners/
 */
@Component
class TemplateInfoOverrides {
    private static final List<String> MOTLEY_COLORS = List.of(
            "yellow", "grey", "dark-grey", "green", "light-green", "dark-green", "blue", "light-blue", "dark-blue",
            "bordo", "white-old",
            "yellow-old", "grey-old");
    private static final List<String> NEW_FOOTER_BACKGROUND = List.of("red", "green", "blue");
    private static final List<String> INSTALL_YES_COUNTER = List.of(
            "https://mc.yandex.ru/watch/37000905?page-url=https%3A%2F%2Fyandex.ru%3Fyclid%3D{LOGID}");
    private static final List<String> CLICK_COUNTER = List.of("https://yabs.yandex.ru/resource/spacer.gif");
    private static final List<String> POPUP_TYPES = List.of("landing", "download", "extensions-landing");
    private static final List<String> TEASER_TYPES = List.of("landing", "download", "extensions-landing",
            "extensions-download");
    private static final List<String> UNDER_SEARCH_TYPES = List.of("landing", "download", "extensions-landing",
            "extensions-download");
    private static final List<String> ARROW_POSITIONS = List.of("center-bottom", "right-top", "left-bottom");
    private static final List<String> PLATFORM = List.of("android", "ios");
    private static final List<String> BUTTON_COLOR = List.of("white", "yellow");
    private static final List<String> T_COLOR = List.of("#ffffff", "#000000");
    private static final List<String> SERVICE = List.of("afisha", "autoru", "avia", "browser", "disk", "drive", "eda",
            "health", "keyboard", "kinopoisk", "launcher", "light-browser", "mail", "maps", "market", "metro",
            "money", "music", "navi", "rabota", "realty", "search", "taxi", "taximeter", "trains", "translate",
            "transport", "uber", "weather", "yango", "yawidget", "zen", "edadeal", "lavka", "pro");
    private static final List<String> PRODUCTS = List.of("uslugi", "districts", "collections", "q", "games", "sport",
            "tutor", "avia", "hotel", "train", "travel", "stream");
    private static final List<String> DESIGN_TYPES = List.of("old-2b", "new-1b", "new-2b", "stripe-big",
            "stripe-default");
    private static final List<String> POPUP_COLORS = List.of("FFCC00", "f9d34c", "fdcb2f", "ffffff", "57ad5e",
            "7bc761", "9ed477", "64842a", "7fad2e", "80b9ff", "a5ceff", "c6e0ff", "35b5de", "1783a1", "5bb9cf",
            "1687ef", "3174be", "4b96e8", "d0c9ff", "e1d8fe", "303030", "b6bac9", "eef2f6", "f45e0f", "ca5518",
            "f1892a", "efa434", "f2f2f2", "343434");

    private static final String COUNTER_FORMAT = "^[0-9a-zA-Z_-]*$";

    private static final List<Restriction> COLOR_FORMAT_RESTRICTION = Restrictions.regexp("^#[0-9a-f]{6}$");

    private static final Map<Long, InternalTemplateOverrides> OVERRIDES_MAP = ImmutableMap.<Long,
                    InternalTemplateOverrides>builder()
            // Тизер на десктопной и планшетной морде
            .put(TEASER_WITH_TIME_TEMPLATE_ID, new InternalTemplateOverrides()
                    .valueRestriction(2334L, Restrictions.teaserImg())
                    .valueRestriction(2335L, List.of(Restrictions.maxTextLength(true, 44))))

            // Тизер 2013 (json)
            .put(TEASER_TEMPLATE_ID, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(3280L, 3282L, 3283L), Set.of(3281L, 3487L))
                    .requiredVariant(Set.of(3280L, 3281L, 3282L), Set.of(3283L, 3487L))
                    .requiredVariant(Set.of(3280L, 3281L), Set.of(3282L, 3283L, 3487L))
                    .requiredVariant(Set.of(3280L, 3282L), Set.of(3281L, 3283L, 3487L))
                    .requiredVariant(Set.of(3280L), Set.of(3281L, 3282L, 3283L, 3487L))
                    // Заголовок (1 строка)
                    .valueRestriction(3280L, Restrictions.teaserHeader())
                    // Заголовок (2 строка)
                    .valueRestriction(3281L, Restrictions.teaserHeader())
                    // Подпись (1 строка)
                    .valueRestriction(3282L, Restrictions.teaserLabel())
                    // Подпись (2 строка)
                    .valueRestriction(3283L, Restrictions.teaserLabel())
                    // Картинка
                    .valueRestriction(3279L, Restrictions.teaserImg())
                    // Картинка для ретины
                    .valueRestriction(3755L, Restrictions.teaserRetinaImg())
                    // Альтернативный текст
                    .valueRestriction(3284L, List.of(Restrictions.maxTextLength(true, 100))))

            // Баннер на Афише
            .put(641L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(2770L, 2772L), Set.of(2771L, 2773L, 3570L))
                    .addHiddenResources(2771L, 2773L, 3570L)
                    // Альтернативный текст
                    .valueRestriction(2769L, List.of(Restrictions.maxTextLength(true, 100)))
                    // Картинка
                    .valueRestriction(2770L, Restrictions.afishaImg()))

            // Маленький тизер
            .put(642L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(), Set.of(2774L, 2775L, 2777L, 2778L, 2779L, 2782L, 2822L))
                    // Подпись (1 строка)
                    .valueRestriction(2776, List.of(Restrictions.maxTextLength(true, 70)))
                    // Картинка
                    .valueRestriction(2780, List.of(
                            Restrictions.imageSizeKbMax(true, DEFAULT_IMAGE_MAX_SIZE_KB),
                            Restrictions.imageDimensionsEq(true, 96, 96),
                            Restrictions.imageFormatIn(true, List.of(PNG))
                    ))
                    // Заголовок (строка 1), Заголовок (строка 2), Подпись (строка 2), Альтернативный текст, ID
                    // Ссылка 2, Счетчик закрытия
                    .addHiddenResources(2774L, 2775L, 2777L, 2778L, 2779L, 2782L, 2822L))

            // Teaser_inline (BSDEV-55306)
            .put(TEASER_INLINE_TEMPLATE_ID, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(3344L, 3346L, 3347L), Set.of(3345L))
                    .requiredVariant(Set.of(3344L, 3345L, 3346L), Set.of(3347L))
                    .requiredVariant(Set.of(3344L, 3345L), Set.of(3346L, 3347L))
                    .requiredVariant(Set.of(3344L, 3346L), Set.of(3345L, 3347L))
                    .requiredVariant(Set.of(3344L), Set.of(3345L, 3346L, 3347L))
                    // Заголовок (1 строка)
                    .valueRestriction(3344, Restrictions.teaserHeader())
                    // Заголовок (2 строка)
                    .valueRestriction(3345, Restrictions.teaserHeader())
                    // Подпись (1 строка)
                    .valueRestriction(3346, Restrictions.teaserLabel())
                    // Подпись (2 строка)
                    .valueRestriction(3347, Restrictions.teaserLabel())
                    // Картинка
                    .valueRestriction(3343, Restrictions.teaserImg())
                    // Картинка для ретины
                    .valueRestriction(3756, Restrictions.teaserRetinaImg()))

            // Teaser_inline (BSDEV-58023)
            .put(TEASER_INLINE_BROWSER_TEMPLATE_ID, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(3675L, 3677L, 3678L), Set.of(3676L, 3693L, 3681L, 3682L, 3688L, 3689L,
                            3690L, 3691L, 3686L, 3684L, 3692L))
                    .requiredVariant(Set.of(3675L, 3676L, 3677L), Set.of(3678L, 3693L, 3681L, 3682L, 3688L, 3689L,
                            3690L, 3691L, 3686L, 3684L, 3692L))
                    .requiredVariant(Set.of(3675L, 3676L), Set.of(3677L, 3678L, 3693L, 3681L, 3682L, 3688L, 3689L,
                            3690L, 3691L, 3686L, 3684L, 3692L))
                    .requiredVariant(Set.of(3675L, 3677L), Set.of(3676L, 3678L, 3693L, 3681L, 3682L, 3688L, 3689L,
                            3690L, 3691L, 3686L, 3684L, 3692L))
                    .requiredVariant(Set.of(3675L), Set.of(3676L, 3677L, 3678L, 3693L, 3681L, 3682L, 3688L, 3689L,
                            3690L, 3691L, 3686L, 3684L, 3692L))
                    // Заголовок (1 строка)
                    .valueRestriction(3675L, Restrictions.teaserHeader())
                    // Заголовок (2 строка)
                    .valueRestriction(3676L, Restrictions.teaserHeader())
                    // Подпись (1 строка)
                    .valueRestriction(3677L, Restrictions.teaserLabel())
                    // Подпись (2 строка)
                    .valueRestriction(3678L, Restrictions.teaserLabel())
                    // Альтернативный текст
                    .valueRestriction(3679L, List.of(Restrictions.maxTextLength(true, 100)))
                    // Картинка
                    .valueRestriction(3674L, Restrictions.teaserImg())
                    // Картинка для ретины
                    .valueRestriction(4143L, Restrictions.teaserRetinaImg())
                    // Тип (type)
                    .addResourceChoices(3680L, List.of("landing", "download", "extensions-landing")))

            // Медийный баннер на таче (BSDEV-53507)
            .put(MEDIA_BANNER_TEMPLATE_ID, new InternalTemplateOverrides()
                    // Высота изображения (bnH)
                    .addResourceChoices(3335, List.of("67"))
                    // Ширина изображения (bnW)
                    .addResourceChoices(3336, List.of("320"))
                    // Картинка (bnImg)
                    .valueRestriction(3337, List.of(
                            Restrictions.imageSizeKbMax(true, 120),
                            Restrictions.imageDimensionsEq(true, 640, 134),
                            Restrictions.imageFormatIn(true, List.of(GIF, JPEG, PNG))
                    ))
                    // Альтернативный текст (bnAlt)
                    .valueRestriction(3338, List.of(Restrictions.maxTextLength(true, 100)))
                    // Счетчик (bnCount), Флаг (gif_tizer), Автор (textauthor)
                    .addHiddenResources(3340L, 3341L, 3468L))

            // Popup (BSDEV-53493)
            .put(848L, new InternalTemplateOverrides()
                    // Кнопка отказа
                    .valueRestriction(3262L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Кнопка согласия
                    .valueRestriction(3263L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Фон
                    .addResourceChoices(3264L, POPUP_COLORS)
                    // Картинка в png
                    .valueRestriction(3265L, List.of(
                            Restrictions.imageDimensionsSquare(true, Set.of(PNG, SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG, GIF))
                    ))
                    // Картинка в svg
                    .valueRestriction(3266L, List.of(
                            Restrictions.imageDimensionsSquare(true, Set.of(PNG, SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG, GIF))
                    ))
                    // Текст
                    .valueRestriction(3269L, List.of(Restrictions.maxTextLength(true, 70)))
                    // Тип
                    .addResourceChoices(3270L, POPUP_TYPES)
                    // Тип дизайна
                    .addResourceChoices(5252L, DESIGN_TYPES)
                    // Цель на установку (goal_yes), Цель на отказ (goal_no), Автор (textauthor),
                    // Инлайн урл, Счетчик кнопки согласия
                    .addHiddenResources(3407L, 3408L, 3465L, 3516L, 3517L))

            // Popup (BSDEV-58023)
            .put(1003L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(3736L, 3737L, 3738L), Set.of(4976L))
                    .requiredVariant(Set.of(4976L), Set.of(3732L, 3733L, 3734L, 3735L, 3736L, 3737L, 3738L))
                    // Кнопка отказа
                    .valueRestriction(3713L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Кнопка согласия
                    .valueRestriction(3714L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Фон
                    .addResourceChoices(3715L, POPUP_COLORS)
                    // Картинка в png
                    .valueRestriction(3716L, List.of(
                            Restrictions.imageDimensionsSquare(true, Set.of(PNG, SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG, GIF))
                    ))
                    // Картинка в svg
                    .valueRestriction(3717L, List.of(
                            Restrictions.imageDimensionsSquare(true, Set.of(PNG, SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG, GIF))
                    ))
                    // Текст
                    .valueRestriction(3720L, List.of(Restrictions.maxTextLength(true, 70)))
                    // Тип
                    .addResourceChoices(3721L, POPUP_TYPES)
                    // Тип дизайна
                    .addResourceChoices(5254L, DESIGN_TYPES)
                    // Цель на установку (goal_yes), Цель на отказ (goal_no), Автор (textauthor),
                    // Инлайн урл, Счетчик кнопки согласия
                    .addHiddenResources(3726L, 3727L, 3728L, 3730L, 3731L))

            // Колдунщик
            .put(858L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(3357L), Set.of()))

            // Цветной тизер (SuperTeaser BSDEV-53505)
            .put(860L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(3571L, 3572L), Set.of(3328L, 3469L))
                    .requiredVariant(Set.of(3328L), Set.of(3571L, 3572L, 3469L))
                    //Текст кнопки (button_text)
                    .valueRestriction(4613L, List.of(Restrictions.maxTextLength(true, 30)))
                    //Цвет кнопки (button_color)
                    .addResourceChoices(3327, BUTTON_COLOR)
                    //Счетчик (counter)
                    .valueRestriction(3329, Restrictions.regexp(COUNTER_FORMAT))
                    //Сервис (service)
                    .addResourceChoices(3330, SERVICE)
                    //Цвет текста (t_color)
                    .addResourceChoices(3331, T_COLOR)
                    //Заголовок (title)
                    .valueRestriction(3333, List.of(Restrictions.maxTextLength(true, 20)))
                    //Текст (text)
                    .valueRestriction(3332, List.of(Restrictions.maxTextLength(true, 57)))
                    //Большая картинка для фона (b_image)
                    .valueRestriction(3571, List.of(
                            Restrictions.imageSizeKbMax(true, 100),
                            Restrictions.imageDimensionsEq(true, 1280, 394),
                            Restrictions.imageFormatIn(true, List.of(JPEG, PNG))
                    ))
                    //Маленькая картинка для фона (s_image)
                    .valueRestriction(3572, List.of(
                            Restrictions.imageSizeKbMax(true, 100),
                            Restrictions.imageDimensionsEq(true, 640, 394),
                            Restrictions.imageFormatIn(true, List.of(JPEG, PNG))
                    ))
                    //Автор (textauthor),
                    .addHiddenResources(3469L))

            // Popup (BSDEV-64587)
            .put(2999L, new InternalTemplateOverrides()
                    // Кнопка отказа
                    .valueRestriction(3794L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Кнопка согласия
                    .valueRestriction(3795L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Текст
                    .valueRestriction(3801L, List.of(Restrictions.maxTextLength(true, 70)))
                    // Тема (motley)
                    .addResourceChoices(4436L, MOTLEY_COLORS))

            // Тизер для ТВ морды
            .put(3000L, new InternalTemplateOverrides()
                    // Кнопка отказа
                    .valueRestriction(3823L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Кнопка согласия
                    .valueRestriction(3824L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Картинка в png
                    .valueRestriction(3826L, List.of(
                            Restrictions.imageDimensionsEq(true, 60, 60, Set.of(PNG)),
                            Restrictions.imageDimensionsSquare(true, Set.of(SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG, GIF))
                    ))
                    // Картинка в svg
                    .valueRestriction(3827L, List.of(
                            Restrictions.imageDimensionsEq(true, 120, 120, Set.of(PNG)),
                            Restrictions.imageDimensionsSquare(true, Set.of(SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(SVG, PNG, GIF))
                    ))
                    // Текст
                    .valueRestriction(3830L, List.of(Restrictions.maxTextLength(true, 70)))
                    // Тип
                    .addResourceChoices(3831L, TEASER_TYPES)
                    // Тема (motley)
                    .addResourceChoices(4437L, MOTLEY_COLORS)
                    // Фон, Канал (channel), Цель на установку (goal_yes), Цель на отказ (goal_no), Автор (textauthor),
                    // Инлайн урл, Счетчик кнопки согласия, Описание ссылки повторить скачивание,
                    // Текст на ссылке повторить скачивание, Текст ЛС (legal.text), Ссылка ЛС (legal.link),
                    // Текст ссылки ЛС (legal.linktext)
                    .addHiddenResources(3825L, 3834L, 3836L, 3837L, 3838L, 3840L, 3841L, 3963L, 3964L, 3965L, 3966L,
                            3967L))

            // Мобильный подвал серпа
            .put(3103L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(4053L, 4054L, 4055L, 4056L, 4057L, 4058L), Set.of())
                    // Платформа
                    .addResourceChoices(4053L, PLATFORM)
                    //Иконка
                    .valueRestriction(4054L, List.of(
                            Restrictions.imageSizeKbMax(true, 10),
                            Restrictions.imageDimensionsEq(true, 72, 72),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG))
                    ))
                    //Иконка для ретины
                    .valueRestriction(4055L, List.of(
                            Restrictions.imageSizeKbMax(true, 10),
                            Restrictions.imageDimensionsEq(true, 144, 144),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG))
                    ))
                    //Заголовок
                    .valueRestriction(4056L, List.of(Restrictions.maxTextLength(true, 25)))
                    //Текст
                    .valueRestriction(4057L, List.of(Restrictions.maxTextLength(true, 44)))
                    //Текст на кнопке
                    .valueRestriction(4058L, List.of(Restrictions.maxTextLength(true, 11))))

            // Popup залогина (BSDEV-66773)
            .put(3125L, new InternalTemplateOverrides()
                    // Текст кнопки отказа
                    .valueRestriction(4148L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Текст кнопки согласия
                    .valueRestriction(4147L, List.of(Restrictions.maxTextLength(true, 15)))
                    // Текст
                    .valueRestriction(4146L, List.of(Restrictions.maxTextLength(true, 70))))

            // Новый футер на десктопном серпе
            .put(3148L, new InternalTemplateOverrides()
                    .requiredVariantErrorMessage("Должно быть заполнено либо \"Картинка\" либо \"Фоновая картинка\"")
                    .requiredVariant(Set.of(4280L), Set.of(4620L))
                    .requiredVariant(Set.of(4620L), Set.of(4280L))
                    // Тип
                    .addResourceChoices(4275L, List.of("landing", "extensions-landing"))
                    // Заголовок
                    .valueRestriction(4276L, List.of(Restrictions.maxTextLength(true, 30)))
                    // Текст
                    .valueRestriction(4277L, List.of(Restrictions.maxTextLength(true, 60)))
                    // Текст на кнопке
                    .valueRestriction(4278L, List.of(Restrictions.maxTextLength(true, 20)))
                    // Картинка
                    .valueRestriction(4280L, List.of(
                            Restrictions.imageFormatIn(true, List.of(SVG)),
                            Restrictions.imageSizeKbMax(true, 20))
                    )
                    // Градиент фона
                    .addResourceChoices(4281L, NEW_FOOTER_BACKGROUND)
                    // Фоновая картинка
                    .valueRestriction(4620L, List.of(
                            Restrictions.imageFormatIn(true, List.of(PNG)),
                            Restrictions.imageDimensionsEq(true, 402, 328))
                    ))

            // Новый футер на десктопном серпе (инлайн)
            .put(3149L, new InternalTemplateOverrides()
                    // Тип
                    .addResourceChoices(4284L, List.of("inline"))
                    // Заголовок
                    .valueRestriction(4285L, List.of(Restrictions.maxTextLength(true, 30)))
                    // Текст
                    .valueRestriction(4286L, List.of(Restrictions.maxTextLength(true, 60)))
                    // Текст на кнопке
                    .valueRestriction(4287L, List.of(Restrictions.maxTextLength(true, 20)))
                    // Картинка
                    .valueRestriction(4289L, Restrictions.seprFooterImg())
                    // Градиент фона
                    .addResourceChoices(4290L, NEW_FOOTER_BACKGROUND)
                    // Счетчик кнопки согласия
                    .addResourceChoices(4304L, INSTALL_YES_COUNTER))

            // Системная полоска на серпе
            .put(3210L, new InternalTemplateOverrides()
                    // Продукт
                    .addResourceChoices(4402L, List.of("set", "sethome"))
                    // Текст
                    .valueRestriction(4403L, List.of(Restrictions.maxTextLength(true, 54))))

            // Тизер для ПП
            .put(3176L, new InternalTemplateOverrides()
                    .requiredVariantErrorMessage("Должно быть заполнено либо \"ссылка для фичи с включением\", либо " +
                            "\"ссылка на лэндинг фичи\"")
                    .requiredVariant(Set.of(4439L), Set.of(4319L))
                    .requiredVariant(Set.of(4319L), Set.of(4439L))
                    // Цвет для кнопки контекстного меню
                    .valueRestriction(4320L, COLOR_FORMAT_RESTRICTION)
                    // Заголовок для карточки в ПП
                    .valueRestriction(4321L, List.of(Restrictions.maxTextLength(true, 30)))
                    // Цвет текста заголовка
                    .valueRestriction(4322L, COLOR_FORMAT_RESTRICTION)
                    // Текст для карточки в ПП
                    .valueRestriction(4323L, List.of(Restrictions.maxTextLength(true, 120)))
                    // Цвет основного текста
                    .valueRestriction(4324L, COLOR_FORMAT_RESTRICTION)
                    // Цвет для кнопки
                    .valueRestriction(4325L, COLOR_FORMAT_RESTRICTION)
                    // Текст для кнопки в ПП
                    .valueRestriction(4326L, List.of(Restrictions.maxTextLength(true, 18)))
                    // Цвет текста кнопки
                    .valueRestriction(4327L, COLOR_FORMAT_RESTRICTION)
                    // Цвет фона карточки
                    .valueRestriction(4328L, COLOR_FORMAT_RESTRICTION)
                    // Ссылка на картинку(dpi_1)
                    .valueRestriction(4329L, List.of(
                            Restrictions.imageFormatIn(true, List.of(PNG)),
                            Restrictions.imageDimensionsEq(true, 360, null))
                    )
                    // Ссылка на картинку(dpi_1.5)
                    .valueRestriction(4330L, List.of(
                            Restrictions.imageFormatIn(true, List.of(PNG)),
                            Restrictions.imageDimensionsEq(true, 540, null))
                    )
                    // Ссылка на картинку(dpi_2)
                    .valueRestriction(4331L, List.of(
                            Restrictions.imageFormatIn(true, List.of(PNG)),
                            Restrictions.imageDimensionsEq(true, 720, null))
                    )
                    // Счетчик кликов
                    .addResourceChoices(4398L, CLICK_COUNTER)
            )

            // Тизер для тв-морды (BSDEV-70467)
            .put(3209L, new InternalTemplateOverrides()
                    // Картинка
                    .valueRestriction(4395, List.of(
                            Restrictions.imageSizeKbMax(true, DEFAULT_IMAGE_MAX_SIZE_KB),
                            Restrictions.imageDimensionsEq(true, 280, 180),
                            Restrictions.imageFormatIn(true, List.of(PNG))
                    ))
                    // Текст
                    .valueRestriction(4396, List.of(Restrictions.maxTextLength(true, 43)))
                    // Счетчик кликов
                    .addResourceChoices(4443, CLICK_COUNTER))

            // Ссылка под поисковой строкой (BSDEV-70253)
            .put(3197L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(4335L, 4337L), Set.of())
                    // Текст
                    .valueRestriction(4333L, List.of(Restrictions.maxTextLength(true, 28)))
                    // Картинка
                    .valueRestriction(4334L, Restrictions.underSearchImg())
                    // Картинка retina
                    .valueRestriction(4335L, Restrictions.underSearchRetinaImg())
                    // Тип
                    .addResourceChoices(4337L, UNDER_SEARCH_TYPES))

            // браузер с прямым скачиванием, морда
            .put(3198L, new InternalTemplateOverrides()
                    // Текст
                    .valueRestriction(4338L, List.of(Restrictions.maxTextLength(true, 28)))
                    // Картинка
                    .valueRestriction(4339L, Restrictions.underSearchImg())
                    // Картинка retina
                    .valueRestriction(4340L, Restrictions.underSearchRetinaImg())
                    // Тип
                    .addResourceChoices(4342L, UNDER_SEARCH_TYPES)
                    // Позиция стрелки для Я.Бро (position)
                    .addResourceChoices(4344L, ARROW_POSITIONS))

            // Блок меню на морде (десктоп (BSDEV-75635)
            .put(3327L, new InternalTemplateOverrides()
                    // Текст
                    .valueRestriction(4956, List.of(Restrictions.maxTextLength(true, 20))))

            // Меню на морде (тач) (BSDEV-75635)
            .put(3328L, new InternalTemplateOverrides()
                    // Заголовок
                    .valueRestriction(4958, List.of(Restrictions.maxTextLength(true, 15)))
                    // Текст
                    .valueRestriction(4959, List.of(Restrictions.maxTextLength(true, 48)))
                    // Картинка (png)
                    .valueRestriction(4960, List.of(
                            Restrictions.imageDimensionsEq(true, 24, 24),
                            Restrictions.imageFormatIn(true, List.of(PNG))
                    ))
                    // Картинка (svg)
                    .valueRestriction(4961, List.of(
                            Restrictions.imageDimensionsEq(true, 24, 24),
                            Restrictions.imageFormatIn(true, List.of(SVG))
                    )))

            // Баннер МКБ для Грибов (BSSERVER-12582)
            .put(3335L, new InternalTemplateOverrides()
                    // Продукт
                    .addResourceChoices(5020, PRODUCTS)
                    // Картинка
                    .valueRestriction(5021, List.of(
                            Restrictions.imageSizeKbMax(true, 120),
                            Restrictions.imageDimensionsEq(true, 240, 400),
                            Restrictions.imageFormatIn(true, List.of(JPEG, PNG, GIF))
                    )))

            // Сервис Яндекса для промоблока серпа (BSDEV-71358)
            .put(3304L, new InternalTemplateOverrides()
                    // Сервис
                    .valueRestriction(4611, List.of(Restrictions.maxTextLength(true, 20))))

            .put(3278L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(4480L, 4481L, 4482L), Set.of(4913L))
                    .requiredVariant(Set.of(4913L), Set.of(4476L, 4477L, 4478L, 4479L, 4480L, 4481L, 4482L))
                    // Кнопка отказа
                    .valueRestriction(4457, List.of(Restrictions.maxTextLength(true, 15)))
                    // Кнопка согласия
                    .valueRestriction(4458, List.of(Restrictions.maxTextLength(true, 15)))
                    // Картинка в png
                    .valueRestriction(4460, List.of(
                            Restrictions.imageDimensionsEq(true, 60, 60, Set.of(PNG)),
                            Restrictions.imageDimensionsSquare(true, Set.of(SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(PNG, SVG, GIF))
                    ))
                    // Картинка в svg
                    .valueRestriction(4461, List.of(
                            Restrictions.imageDimensionsEq(true, 120, 120, Set.of(PNG)),
                            Restrictions.imageDimensionsSquare(true, Set.of(SVG, GIF)),
                            Restrictions.imageFormatIn(true, List.of(SVG, PNG, GIF))
                    ))
                    // Текст
                    .valueRestriction(4464, List.of(Restrictions.maxTextLength(true, 70)))
                    // Тип
                    .addResourceChoices(4465, TEASER_TYPES)
                    // Фон, Паранджа, Канал (channel), Цель на установку (goal_yes), Цель на отказ (goal_no),
                    // Автор (textauthor), Инлайн урл, Счетчик кнопки согласия, Вариант анимации появления элементов,
                    // true/false (показывать баннер при возвращении на сервис или нет), Текст в окне предустановки
                    // Токен для окна предустановки, Ссылка на Соглашение в окне предустановки,
                    // Текст Соглашения в окне предустановки, Текст 2 в окне предустановки, Токен 2 для окна
                    // предустановки
                    .addHiddenResources(4459L, 4466L, 4468L, 4470L, 4471L, 4472L, 4474L, 4475L, 4543L, 4544L, 4545L,
                            4546L, 4547L, 4548L, 4549L, 4550L))

            .put(3338L, new InternalTemplateOverrides()
                    .requiredVariant(Set.of(), Set.of(5210L, 5212L, 5214L))
                    .requiredVariant(Set.of(5210L), Set.of(5212L, 5214L))
                    .requiredVariant(Set.of(5210L, 5212L), Set.of(5214L))
                    .requiredVariant(Set.of(5210L, 5214L), Set.of(5212L))
                    .requiredVariant(Set.of(5210L, 5212L, 5214L), Set.of())

                    // Иконка приложения
                    .valueRestriction(5203, List.of(
                            Restrictions.imageDimensionsEq(true, 28, 28),
                            Restrictions.imageFormatIn(true, List.of(SVG, PNG))
                    ))
                    // Расположение иконки
                    .addResourceChoices(5204, List.of("in", "out"))
                    // Заголовок
                    .valueRestriction(5205, List.of(Restrictions.maxTextLength(true, 40)))
                    // Иконка рейтинга
                    .valueRestriction(5207, List.of(
                            Restrictions.imageFormatIn(true, List.of(SVG)),
                            Restrictions.imageSizeKbMax(true, 2)
                    ))
                    // Иконка для подписи 1
                    .valueRestriction(5209, List.of(
                            Restrictions.imageFormatIn(true, List.of(SVG)),
                            Restrictions.imageSizeKbMax(true, 2)
                    ))
                    // Текст подписи 1
                    .valueRestriction(5210, List.of(Restrictions.maxTextLength(true, 22)))
                    // Иконка для подписи 2
                    .valueRestriction(5211, List.of(
                            Restrictions.imageFormatIn(true, List.of(SVG)),
                            Restrictions.imageSizeKbMax(true, 2)
                    ))
                    // Текст подписи 2
                    .valueRestriction(5212, List.of(Restrictions.maxTextLength(true, 22)))
                    // Иконка для подписи 3
                    .valueRestriction(5213, List.of(
                            Restrictions.imageFormatIn(true, List.of(SVG)),
                            Restrictions.imageSizeKbMax(true, 2)
                    ))
                    // Текст подписи 3
                    .valueRestriction(5214, List.of(Restrictions.maxTextLength(true, 22))))

            // Десктопный баннер на Морде (Internal)
            .put(DESKTOP_BANNER_ON_MORDA_TEMPLATE_ID, new InternalTemplateOverrides()
                    // Alt текст
                    .valueRestriction(5248, List.of(Restrictions.maxTextLength(true, 100)))
                    // Картинка (hidpi)
                    .valueRestriction(5249, List.of(
                            Restrictions.imageDimensionsEq(true, 1456, 180),
                            Restrictions.imageSizeKbMax(true, 400),
                            Restrictions.imageFormatIn(true, List.of(GIF, JPEG, PNG))
                    )))

            // Баннер на 404
            .put(BANNER_ON_404_TEMPLATE_ID, new InternalTemplateOverrides()
                    // Alt текст
                    .valueRestriction(5730, List.of(Restrictions.maxTextLength(true, 100)))
                    // Картинка (hidpi)
                    .valueRestriction(5731, List.of(
                            Restrictions.imageDimensionsEq(true, 336, 280),
                            Restrictions.imageSizeKbMax(true, 150),
                            Restrictions.imageFormatIn(true, List.of(GIF, JPEG, PNG))
                    )))

            .build();

    /**
     * Получить переопределения для шаблона {@code templateId}
     */
    Optional<InternalTemplateOverrides> getOverrides(Long templateId) {
        return Optional.ofNullable(OVERRIDES_MAP.get(templateId));
    }
}
