package ru.yandex.direct.core.entity.internalads.service;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ReadOnlyDirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.ResourceChoice;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceRestriction;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.model.TemplateResource;
import ru.yandex.direct.core.entity.internalads.restriction.Restriction;
import ru.yandex.direct.core.entity.internalads.restriction.Restrictions;
import ru.yandex.direct.utils.FunctionalUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Map.entry;

/**
 * Сервис, предоставляющий информацию о шаблоне внутренней рекламы.
 */
@Service
@ParametersAreNonnullByDefault
public class TemplateInfoService {

    // отображаемые значения для ресурса Счетчик закрытия
    private static final String CLOSE_BY_CAMPAIGN_COUNTER_DISPLAY_VALUE = "RF кампании";
    private static final String CLOSE_BY_AD_GROUP_COUNTER_DISPLAY_VALUE = "RF группы";

    public static final String DEFAULT_RESOURCE_RESTRICTIONS_ERROR_MESSAGE = "Обязательные поля заполнены неправильно";

    private static final List<String> AGE_VALUES = List.of("0+", "6+", "12+", "16+", "18+");

    private static final List<ResourceChoice> AGE_CHOICES = FunctionalUtils.mapList(AGE_VALUES, ResourceChoice::from);

    private static final List<ResourceChoice> CLOSE_COUNTER_CHOICES = List.of(
            ResourceChoice.from(CLOSE_BY_CAMPAIGN_COUNTER_DISPLAY_VALUE, Constants.CLOSE_BY_CAMPAIGN_COUNTER_VALUE),
            ResourceChoice.from(CLOSE_BY_AD_GROUP_COUNTER_DISPLAY_VALUE, Constants.CLOSE_BY_AD_GROUP_COUNTER_VALUE));

    private static final Map<ResourceType, List<ResourceChoice>> RESOURCE_CHOICES_BY_TYPE =
            Map.ofEntries(
                    entry(ResourceType.AGE, AGE_CHOICES),
                    entry(ResourceType.CLOSE_COUNTER, CLOSE_COUNTER_CHOICES)
            );

    private final TemplateResourceService templateResourceService;
    private final TemplateInfoOverrides templateOverrides;

    TemplateInfoService(TemplateResourceService templateResourceService,
            TemplateInfoOverrides templateOverrides) {
        this.templateResourceService = templateResourceService;
        this.templateOverrides = templateOverrides;
    }

    @Nullable
    public InternalTemplateInfo getByTemplateId(Long templateId) {
        List<InternalTemplateInfo> templateInfos = getByTemplateIds(Set.of(templateId));
        return templateInfos.isEmpty() ? null : templateInfos.get(0);
    }

    /**
     * Получить информацию по указанным шаблонам внутренней рекламы.
     * <p>
     * Информация о шаблоне включает состав и описание ресурсов шаблона, ресурсы обязательные для заполнения,
     * ограничения на значения ресурсов.
     * <p>
     * Ресурсы выдаются в том порядке, в котором должны быть показаны на форме пользователю
     */
    public List<InternalTemplateInfo> getByTemplateIds(Collection<Long> templateIds) {
        if (templateIds.isEmpty()) {
            return emptyList();
        }
        return EntryStream.of(getTemplateInfoByIds(templateIds))
                .mapKeys(templateOverrides::getOverrides)
                .mapKeyValue((maybeOverrides, templateInfo) -> maybeOverrides
                        .map(o -> o.applyTo(templateInfo))
                        .orElse(templateInfo))
                .toList();
    }

    private Map<Long, InternalTemplateInfo> getTemplateInfoByIds(Collection<Long> templateIds) {
        var resources = templateResourceService.getReadonlyByTemplateIds(templateIds);
        var resourcesByTemplateId = EntryStream.of(resources)
                .values()
                .mapToEntry(ReadOnlyDirectTemplateResource::getTemplateId, Function.identity())
                .grouping();
        return EntryStream.of(resourcesByTemplateId)
                .mapToValue(this::createTemplateInfo)
                .toMap();
    }

    /**
     * @return информацию о шаблоне с ресурсами; ресурсы уже выдаются в том порядке, который должен быть показан
     * пользователю
     */
    private InternalTemplateInfo createTemplateInfo(Long templateId,
            List<? extends ReadOnlyDirectTemplateResource> templateResources) {
        var orderedResources = StreamEx.of(templateResources)
                .sorted(Comparator.comparing(ReadOnlyDirectTemplateResource::getPosition))
                .toList();

        var resourceInfoList = StreamEx.of(orderedResources)
                .map(this::createResourceInfo)
                .toImmutableList();

        var resourceRestrictions = createResourceRestrictionsRequiredAbsent(orderedResources);

        return new InternalTemplateInfo()
                .withTemplateId(templateId)
                .withResources(resourceInfoList)
                .withResourceRestrictionsErrorMessage(DEFAULT_RESOURCE_RESTRICTIONS_ERROR_MESSAGE)
                .withResourceRestrictions(List.of(resourceRestrictions));
    }

    /**
     * Создать ограничения на состав ресурсов: ресурсы, значения которых должны обязательно присутствовать, и
     * значения, которые должны отсутствовать.
     */
    private ResourceRestriction createResourceRestrictionsRequiredAbsent(
            List<? extends ReadOnlyDirectTemplateResource> orderedResources) {
        Set<Long> requiredResources = StreamEx.of(orderedResources)
                .filter(ReadOnlyDirectTemplateResource::isRequired)
                .map(ReadOnlyDirectTemplateResource::getId)
                .toImmutableSet();
        return new ResourceRestriction()
                .withRequired(requiredResources)
                .withAbsent(emptySet());
    }

    /**
     * Создать описание ресурса {@link ResourceInfo} по описанию {@link TemplateResource}, определяя в качестве
     * валидационных ограничений только
     * дефолтные ограничения по типу ресурса.
     * <p>
     * Здесь не учитывается информация о шаблоне и никакие переопределения валидации, заданные для шаблонов
     * {@link TemplateInfoOverrides}
     *
     * @param templateResource описание ресурса {@link TemplateResource}
     * @return описание ресурса {@link ResourceInfo}
     */
    private ResourceInfo createResourceInfo(ReadOnlyDirectTemplateResource templateResource) {
        TypeWithRestrictions typeWithRestrictions = calcResourceTypeWithRestrictions(templateResource);
        return new ResourceInfo()
                .withId(templateResource.getId())
                .withNo(templateResource.getResourceNo())
                .withPosition(templateResource.getPosition())
                .withLabel(templateResource.getDescription())
                .withType(typeWithRestrictions.resourceType)
                .withValueRestrictions(typeWithRestrictions.restrictions)
                .withChoices(RESOURCE_CHOICES_BY_TYPE.get(typeWithRestrictions.resourceType));
    }

    private static TypeWithRestrictions calcResourceTypeWithRestrictions(
            ReadOnlyDirectTemplateResource templateResource) {
        if (templateResource.isBananaImage()) {
            return new TypeWithRestrictions(ResourceType.IMAGE, Restrictions.defaultImage());
        } else if (templateResource.isBananaUrl()) {
            return new TypeWithRestrictions(ResourceType.URL, Restrictions.defaultUrl());
        } else if (templateResource.isAgeLabel()) {
            return new TypeWithRestrictions(ResourceType.AGE, Restrictions.defaultAge());
        } else if (templateResource.isCloseCounter()) {
            return new TypeWithRestrictions(ResourceType.CLOSE_COUNTER, Restrictions.defaultCloseCounter());
        } else {
            return new TypeWithRestrictions(ResourceType.TEXT, Restrictions.defaultText());
        }
    }

    private static class TypeWithRestrictions {
        private final ResourceType resourceType;
        private final List<Restriction> restrictions;

        private TypeWithRestrictions(ResourceType resourceType, List<Restriction> restrictions) {
            this.resourceType = resourceType;
            this.restrictions = restrictions;
        }
    }
}
