package ru.yandex.direct.core.entity.internalads.service;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import org.apache.commons.collections4.ListUtils;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplatePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplatePlaceRepository;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Сервис для получения связи места с шаблоном для внутренней рекламы
 */
@Service
public class TemplatePlaceService {
    private final TemplatePlaceRepository templatePlaceRepository;
    private final DirectTemplatePlaceRepository directTemplatePlaceRepository;

    public TemplatePlaceService(TemplatePlaceRepository templatePlaceRepository,
            DirectTemplatePlaceRepository directTemplatePlaceRepository) {
        this.templatePlaceRepository = templatePlaceRepository;
        this.directTemplatePlaceRepository = directTemplatePlaceRepository;
    }

    public List<TemplatePlace> getVisibleTemplatesByPlaceIds(@Nullable Set<Long> placeIds) {
        List<TemplatePlace> templatePlaces = getByPlaceIds(placeIds);
        return filterList(
                templatePlaces, templatePlace -> !Constants.INVALID_TEMPLATES.contains(templatePlace.getTemplateId()));
    }

    /**
     * Получить записи таблиц TemplatePlace и DirectTemplatePlace для мест {@code placeIds}
     *
     * @param placeIds — id площадок. Фронт может запрашивать null вместо множества id
     * @return упорядоченный по id площадок и id шаблонов список связок площадок с шаблонами
     */
    public List<TemplatePlace> getByPlaceIds(@Nullable Set<Long> placeIds) {
        if (placeIds == null) {
            var result = ListUtils.sum(
                templatePlaceRepository.getAll(),
                directTemplatePlaceRepository.getAll()
            );
            sortPlaces(result);
            return result;
        }

        if (placeIds.isEmpty()) {
            return List.of();
        }

        var result = ListUtils.sum(
            templatePlaceRepository.getByPlaceIds(placeIds),
            directTemplatePlaceRepository.getByPlaceIds(placeIds)
        );
        sortPlaces(result);
        return result;
    }

    private void sortPlaces(List<TemplatePlace> places) {
        places.sort((left, right) ->
                Objects.equals(left.getPlaceId(), right.getPlaceId()) ?
                        left.getTemplateId().compareTo(right.getTemplateId())
                        : left.getPlaceId().compareTo(right.getPlaceId())
        );
    }
}
