package ru.yandex.direct.core.entity.internalads.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResourceWrapper;
import ru.yandex.direct.core.entity.internalads.model.ReadOnlyDirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplateResourceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplateResourceRepository;

/**
 * Сервис для получения информации о ресурсах шаблонов для внутренней рекламы с учётом единого формата
 */
@Service
@ParametersAreNonnullByDefault
public class TemplateResourceService {
    private final DirectTemplateResourceRepository directTemplateResourceRepository;
    private final TemplateResourceRepository templateResourceRepository;

    static final long DIRECT_TEMPLATE_RESOURCE_STARTING_ID = 1_000_000L;

    public TemplateResourceService(DirectTemplateResourceRepository directTemplateResourceRepository,
            TemplateResourceRepository templateResourceRepository) {
        this.directTemplateResourceRepository = directTemplateResourceRepository;
        this.templateResourceRepository = templateResourceRepository;
    }

    /**
     * Получить ресурсы шаблонов, дополненные описанием и типом от ресурсов единого шаблона
     * и позицией по изначальному ресурсу или автосгенерированной
     * @param templateIds — id шаблонов
     * @return словарь ресурсов с описанием и позицией по их id
     */
    private Map<Long, DirectTemplateResourceWrapper> getByTemplateIds(Collection<Long> templateIds) {
        if (templateIds.isEmpty()) {
            return Map.of();
        }

        return wrapResources(directTemplateResourceRepository.getByTemplateIds(templateIds));
    }

    /**
     * Получить ресурсы шаблонов, дополненные типом от ресурсов единого шаблона
     * и позицией и описанием по изначальному ресурсу или автосгенерированной позицией и описанием от единого ресурса
     * @param resourceIds — id ресурсов
     * @return словарь ресурсов с описанием и позицией по их id
     */
    private Map<Long, DirectTemplateResourceWrapper> getByIds(Collection<Long> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Map.of();
        }

        return wrapResources(directTemplateResourceRepository.getByIds(resourceIds));
    }

    Map<Long, DirectTemplateResourceWrapper> wrapResources(Collection<DirectTemplateResource> resources) {
        var oldResourceIdsToFetch = new HashSet<Long>();
        for (var resource : resources) {
            oldResourceIdsToFetch.add(resource.getUnifiedTemplateResourceId());
            if (resource.getDirectTemplateResourceId() < DIRECT_TEMPLATE_RESOURCE_STARTING_ID) {
                oldResourceIdsToFetch.add(resource.getDirectTemplateResourceId());
            }
        }
        var oldResources = StreamEx.of(templateResourceRepository.getByIds(oldResourceIdsToFetch))
                .mapToEntry(ReadOnlyDirectTemplateResource::getId, r -> r)
                .toMap();

        return StreamEx.of(resources)
                .mapToEntry(DirectTemplateResource::getDirectTemplateResourceId,
                        r -> new DirectTemplateResourceWrapper()
                                .withResource(r)
                                .withDescription(getDescription(r,
                                        oldResources.get(r.getUnifiedTemplateResourceId()),
                                        oldResources.get(r.getDirectTemplateResourceId()))
                                )
                                .withPosition(getPosition(r, oldResources.get(r.getDirectTemplateResourceId())))
                                .withResourceType(oldResources.get(r.getUnifiedTemplateResourceId()).getResourceType()))
                .toMap();
    }

    private String getDescription(DirectTemplateResource resource,
            ReadOnlyDirectTemplateResource unifiedResource,
            @Nullable ReadOnlyDirectTemplateResource oldResource) {
        if (resource.getDirectTemplateResourceId() >= DIRECT_TEMPLATE_RESOURCE_STARTING_ID
                || oldResource == null) {
            return unifiedResource.getDescription();
        }
        return oldResource.getDescription();
    }

    private long getPosition(DirectTemplateResource resource, @Nullable ReadOnlyDirectTemplateResource oldResource) {
        if (resource.getDirectTemplateResourceId() >= DIRECT_TEMPLATE_RESOURCE_STARTING_ID
        || oldResource == null) {
            return resource.getDirectTemplateResourceId();
        }
        return oldResource.getPosition();
    }

    /**
     * Получить ресурсы по их id, включая те ресурсы, которые не были переведены на единый шаблон.
     * Возвращает ресурсы в пригодном только для чтения формате.
     * @param resourceIds — id ресурсов
     * @return словарь ресурсов по их id
     */
    public Map<Long, ReadOnlyDirectTemplateResource> getReadonlyByIds(Collection<Long> resourceIds) {
        var result = new HashMap<Long, ReadOnlyDirectTemplateResource>(getByIds(resourceIds));
        var templateResources = templateResourceRepository.getByIds(resourceIds);
        templateResources.forEach(r -> result.putIfAbsent(r.getId(), r));
        return result;
    }

    /**
     * Получить ресурсы по id шаблонов, включая те ресурсы, которые не были переведены на единый шаблон.
     * Возвращает ресурсы в пригодном только для чтения формате.
     * @param templateIds — id шаблонов
     * @return словарь ресурсов по их id
     */
    public Map<Long, ReadOnlyDirectTemplateResource> getReadonlyByTemplateIds(Collection<Long> templateIds) {
        var result = new HashMap<Long, ReadOnlyDirectTemplateResource>(getByTemplateIds(templateIds));
        var templateResources = templateResourceRepository.getByTemplateIds(templateIds);
        templateResources.forEach(r -> result.putIfAbsent(r.getId(), r));
        return result;
    }

    /**
     * Оставляет только id ресурсов от переданных id шаблонов, которые должны содержать картинку.
     * Работает как с новыми, так и со старыми непереведёнными ресурсами.
     * @param templateIds — id шаблонов, ресурсы которых нужно фильтровать
     * @return множество id ресурсов, которые должны содержать картинку
     */
    public Set<Long> getImageResourceIdsByTemplateIds(Collection<Long> templateIds) {
        if (templateIds.isEmpty()) {
            return Set.of();
        }

        return  EntryStream.of(getReadonlyByTemplateIds(templateIds))
                .filterValues(ReadOnlyDirectTemplateResource::isBananaImage)
                .keys()
                .toSet();
    }

    /**
     * Оставляет только id ресурсов от переданных id ресурсов, которые должны содержать картинку.
     * Работает как с новыми, так и со старыми непереведёнными ресурсами.
     * @param resourceIds — id ресурсов которые нужно фильтровать
     * @return множество id ресурсов, которые должны содержать картинку
     */
    public Set<Long> getImageResourceIdsByResourceIds(Collection<Long> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Set.of();
        }

        return EntryStream.of(getReadonlyByIds(resourceIds))
                .filterValues(ReadOnlyDirectTemplateResource::isBananaImage)
                .keys()
                .toSet();
    }
}
