package ru.yandex.direct.core.entity.inventori.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.adgroup.model.PageGroupTagEnum;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.core.entity.adgroup.model.TargetTagEnum;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.model.StatusModerateBannerPage;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.core.entity.bidmodifier.BannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForForecast;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandSafetyService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.inventori.service.type.AdGroupData;
import ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter;
import ru.yandex.direct.core.entity.inventori.service.type.FrontendData;
import ru.yandex.direct.core.entity.pricepackage.model.BbKeyword;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.ProjectParamCondition;
import ru.yandex.direct.core.entity.pricepackage.model.ProjectParamConjunction;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.projectparam.repository.ProjectParamConditionRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.inventori.model.request.CampaignParametersCorrections;
import ru.yandex.direct.inventori.model.request.MainPageTrafficType;
import ru.yandex.direct.inventori.model.request.ProjectParameter;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.request.TrafficTypeCorrections;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_OUTDOOR;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerAdGroupIdFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.clientIdAndAdGroupIdsFilter;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIdsFilter;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


/**
 * Класс для сборки данных по кампании
 * для отправки в cpm-прогнозатор
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-82223">DIRECT-82223</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-83220">DIRECT-83220</a>
 * <p>
 * Данные отправляются в разбивке по группам.
 * Данные каждой группы конвертируются в inventori таргетинг с помощью {@link AdGroupDataConverter}.
 * <p>
 * Данные не-outdoor групп, имеющие условие "профиль пользователя":
 * список id регионов, список минус-площадок, список форматов баннеров,
 * список сегментов крипты, список сегментов я.аудиторий, корректировки ставок по типу устройств
 * <p>
 * Данные outdoor групп:
 * список форматов видео креативов, список пейдж блоков (оператор + список щитов)
 */
@Service
@ParametersAreNonnullByDefault
public class CampaignInfoCollector {
    private static final Set<AdGroupType> DUAL_MODERATION_GROUP_TYPES = ImmutableSet
            .of(CPM_OUTDOOR);
    static final String INVENTORI_TARGET_TAGS_NAME = "inventori_target_tags";

    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerRepository bannerRepository;
    private final CreativeRepository creativeRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final BidModifierRepository bidModifierRepository;
    private final BidModifierService bidModifierService;
    private final ModerateBannerPagesRepository moderateBannerPagesRepository;
    private final PricePackageRepository pricePackageRepository;
    private final CampaignWithBrandSafetyService campaignWithBrandSafetyService;
    private final ProjectParamConditionRepository projectParamConditionRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public CampaignInfoCollector(
            CampaignRepository campaignRepository,
            CampaignTypedRepository campaignTypedRepository,
            AdGroupRepository adGroupRepository,
            BannerRepository bannerRepository,
            CreativeRepository creativeRepository,
            RetargetingConditionRepository retargetingConditionRepository,
            CryptaSegmentRepository cryptaSegmentRepository,
            BidModifierRepository bidModifierRepository,
            BidModifierService bidModifierService,
            ModerateBannerPagesRepository moderateBannerPagesRepository,
            PricePackageRepository pricePackageRepository,
            CampaignWithBrandSafetyService campaignWithBrandSafetyService,
            ProjectParamConditionRepository projectParamConditionRepository,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerRepository = bannerRepository;
        this.creativeRepository = creativeRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.bidModifierRepository = bidModifierRepository;
        this.bidModifierService = bidModifierService;
        this.moderateBannerPagesRepository = moderateBannerPagesRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.campaignWithBrandSafetyService = campaignWithBrandSafetyService;
        this.projectParamConditionRepository = projectParamConditionRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }


    /**
     * Функция, возвращающая искомые данные по id кампаний
     *
     * @param frontendData     данные фронта
     * @param fromIntapi       происходит ли вызов метода из intapi
     * @param operatorUid      uid оператора
     * @param clientId         id клиента
     * @param campaignIds      id кампаний
     * @param includeNotActive нужно ли учитывать группы с баннерами, которые не видны в БК, в списке таргетов
     *                         сейчас true, только при вызове из intapi
     * @return мапа id кампании -> пара (Список таргетов в разбивке по группам, содержится ли в таргетах группа
     * с ключевыми словами)
     */
    public Map<Long, Pair<List<Target>, Boolean>> collectCampaignsInfoWithClientIdAndUid(
            @Nullable FrontendData frontendData,
            boolean fromIntapi,
            int shard,
            @Nullable Long operatorUid,
            @Nullable ClientId clientId,
            List<Long> campaignIds,
            boolean includeNotActive,
            @Nullable Map<Long, TrafficTypeCorrections> trafficTypeCorrectionsByCampaignIdParam) {
        checkState(fromIntapi || (clientId != null && operatorUid != null));
        campaignIds = campaignIds.stream().filter(Objects::nonNull).collect(toList());
        if (campaignIds.isEmpty()) {
            campaignIds = singletonList(0L);
        }
        Set<Long> adGroupIds;
        if (frontendData == null) {
            adGroupIds = StreamEx.of(
                    adGroupRepository.getAdGroupIdsBySelectionCriteria(shard,
                            new AdGroupsSelectionCriteria().withCampaignIds(StreamEx.of(campaignIds).toSet()),
                            maxLimited()))
                    .toSet();
        } else {
            adGroupIds = singleton(frontendData.getGroupId());
        }
        List<AdGroup> adGroupsAll = adGroupRepository.getAdGroups(shard, adGroupIds);

        List<CampaignWithPricePackage> priceCampaigns = campaignTypedRepository.getSafely(shard,
                campaignIdsFilter(campaignIds),
                CampaignWithPricePackage.class);
        Map<Long, CampaignWithPricePackage> campaignsByCampaignId =
                StreamEx.of(priceCampaigns.stream()).toMap(CampaignWithPricePackage::getId, identity());
        Set<Long> pricePackageIds = priceCampaigns.stream()
                .map(CampaignWithPricePackage::getPricePackageId)
                .collect(Collectors.toSet());
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        Set<Long> pricePackagesForBooking = getPricePackageIdsForBooking(pricePackages.values());
        Set<Long> cidsToForceIncludeGroups = priceCampaigns.stream()
                .filter(campaign -> pricePackagesForBooking.contains(campaign.getPricePackageId()))
                .map(CampaignWithPricePackage::getId)
                .collect(Collectors.toSet());

        // фильтрация активных групп, баннеров и пейджей
        AdGroupsFilteringResult adGroupsFilteringResult = filterActiveAdGroups(shard, adGroupsAll, includeNotActive,
                cidsToForceIncludeGroups);
        Map<Long, Set<Long>> moderatedBannerIdsByActiveAdGroupIds =
                adGroupsFilteringResult.getModeratedBannerIdsByActiveAdGroupIds();

        Set<AdGroup> moderatedAdGroups = StreamEx.of(adGroupsAll)
                .filter(adGroup ->
                        frontendData != null || moderatedBannerIdsByActiveAdGroupIds.containsKey(adGroup.getId()))
                .toSet();
        Set<Long> adGroupIdsFiltered = StreamEx.of(moderatedAdGroups).map(AdGroup::getId).toSet();
        // словари
        Map<Long, Goal> cryptaSegments = cryptaSegmentRepository.getAll();

        // данные группы
        Map<Long, List<Long>> bannerIdsByAdGroupId;
        Map<Long, List<RetargetingCondition>> retargetingConditionsByAdGroupId;

        List<BidModifier> bidModifiersAll;
        if (fromIntapi) {

            bannerIdsByAdGroupId =
                    bannerRepository.type
                            .getSafely(shard, bannerAdGroupIdFilter(adGroupIdsFiltered), BannerWithAdGroupId.class)
                            .stream()
                            .filter(banner ->
                                    moderatedBannerIdsByActiveAdGroupIds.getOrDefault(banner.getAdGroupId(), emptySet())
                                            .contains(banner.getId()))
                            .collect(groupingBy(BannerWithAdGroupId::getAdGroupId,
                                    mapping(BannerWithAdGroupId::getId, toList())));

            retargetingConditionsByAdGroupId =
                    retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, adGroupIdsFiltered);
            bidModifiersAll = !adGroupIdsFiltered.isEmpty() ?
                    bidModifierService.getByAdGroupIds(shard, adGroupIdsFiltered,
                            ImmutableSet.of(BidModifierType.DESKTOP_MULTIPLIER, BidModifierType.MOBILE_MULTIPLIER,
                                    BidModifierType.DEMOGRAPHY_MULTIPLIER),
                            ImmutableSet.of(BidModifierLevel.ADGROUP))
                    : emptyList();
        } else {
            bannerIdsByAdGroupId =
                    bannerRepository.type
                            .getSafely(shard, clientIdAndAdGroupIdsFilter(clientId, adGroupIdsFiltered),
                                    BannerWithAdGroupId.class)
                            .stream()
                            .filter(banner ->
                                    moderatedBannerIdsByActiveAdGroupIds.getOrDefault(banner.getAdGroupId(), emptySet())
                                            .contains(banner.getId()))
                            .collect(groupingBy(BannerWithAdGroupId::getAdGroupId,
                                    mapping(BannerWithAdGroupId::getId, toList())));

            retargetingConditionsByAdGroupId =
                    retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, clientId, adGroupIdsFiltered);
            bidModifiersAll = !adGroupIdsFiltered.isEmpty() ?
                    bidModifierService.getByAdGroupIds(clientId, adGroupIdsFiltered, ImmutableSet.copyOf(campaignIds),
                            ImmutableSet.of(BidModifierType.DESKTOP_MULTIPLIER, BidModifierType.MOBILE_MULTIPLIER,
                                    BidModifierType.DEMOGRAPHY_MULTIPLIER),
                            ImmutableSet.of(BidModifierLevel.ADGROUP), operatorUid)
                    : emptyList();
        }

        // данные баннера
        List<Long> bannerIds = StreamEx.of(bannerIdsByAdGroupId.values())
                .flatMap(Collection::stream)
                .toList();

        Map<Long, Creative> creativesByBannerId = getCreativesByBannerId(shard, clientId, bannerIds);

        if (frontendData != null) {
            frontendData.withCreatives(ifNotNull(frontendData.getVideoCreativeIds(),
                    ids -> creativeRepository.getCreatives(shard, ids)));
        }

        Map<Long, List<BidModifier>> bidModifiersByCampaignId = StreamEx.of(bidModifiersAll)
                .mapToEntry(BidModifier::getCampaignId, identity())
                .grouping();

        Map<Long, CampaignForForecast> campaignByCampaignId = getCampaignByCampaignId(shard, clientId, campaignIds);

        Map<Long, Set<FrontpageCampaignShowType>> frontpageTypesByCampaignId =
                campaignRepository.getFrontpageTypesForCampaigns(shard, campaignIds);
        Map<Long, MainPageTrafficType> mainPageTrafficTypeByCampaignId = collectMainPageTrafficType(frontpageTypesByCampaignId);

        var brandSafetyRetargetingsByCampaignId = campaignWithBrandSafetyService.getRetargetingConditions(campaignIds);

        Map<Long, List<AdGroup>> filteredAdGroupsByCampaignId = StreamEx.of(moderatedAdGroups)
                .mapToEntry(AdGroup::getCampaignId, identity())
                .grouping();

        Map<Long, TrafficTypeCorrections> trafficTypeCorrectionsByCampaignId =
                trafficTypeCorrectionsByCampaignIdParam != null ?
                        trafficTypeCorrectionsByCampaignIdParam : collectCampaignsCorrections(shard, campaignIds);

        // Добываем значение Priority для  групп
        List<Long> allAdGroupIds = filteredAdGroupsByCampaignId.values()
                .stream()
                .flatMap(Collection::stream)
                .map(AdGroup::getId)
                .distinct()
                .collect(toList());
        Map<Long, Long> adGroupPriorities = adGroupRepository.getAdGroupsPriority(shard, allAdGroupIds);

        return StreamEx.of(campaignIds)
                .mapToEntry(identity(), campaignId -> {
                    List<AdGroup> adGroupsForCampaign =
                            filteredAdGroupsByCampaignId.getOrDefault(campaignId, emptyList());
                    var cpmPriceCampaign = campaignsByCampaignId.get(campaignId);
                    return collectCampaignInfo(frontendData, campaignId, adGroupsForCampaign,
                            creativesByBannerId,
                            bannerIdsByAdGroupId, retargetingConditionsByAdGroupId, brandSafetyRetargetingsByCampaignId,
                            adGroupsFilteringResult.getPageBlocksByAdGroupIdFiltered(),
                            adGroupsFilteringResult.getPageExcludedBlocksByAdGroupIdFiltered(),
                            bidModifiersByCampaignId, campaignByCampaignId, mainPageTrafficTypeByCampaignId,
                            frontpageTypesByCampaignId,
                            cryptaSegments,
                            trafficTypeCorrectionsByCampaignId,
                            adGroupPriorities,
                            (cpmPriceCampaign != null) ? pricePackages.get(cpmPriceCampaign.getPricePackageId()) : null);
                })
                .toMap();
    }

    /**
     * Фильтрует группы, во-первых, по активным баннерам, во-вторых, по баннерам, прошедшим второй этап модерации.
     * Группы кампаний из cidsToForceIncludeGroups включаем, даже если нет активных баннеров.
     * На выходе имеем список промодерированных пейджей и список промодерированных баннеров, сгруппированных по
     * AdGroupId, в которых есть промодерированные пейджи/баннеры.
     */
    public AdGroupsFilteringResult filterActiveAdGroups(int shard, Collection<AdGroup> adGroups,
                                                        boolean includeNotActive,
                                                        Set<Long> cidsToForceIncludeGroups) {
        // Берём активные группы и выбираем в них только активные баннеры
        Map<Long, Set<Long>> activeBannerIdsByActiveAdGroupIds = getActiveAdGroupsWithActiveBanners(
                shard, adGroups, includeNotActive, cidsToForceIncludeGroups);
        Set<AdGroup> activeAdGroups = StreamEx.of(adGroups)
                .filter(adGroup -> activeBannerIdsByActiveAdGroupIds.containsKey(adGroup.getId()))
                .toSet();

        // Дополнительная проверка для баннеров с двухэтапной модерацией (например outdoor)
        Set<AdGroupType> dualModerationGroupTypes = includeNotActive ? emptySet() : DUAL_MODERATION_GROUP_TYPES;
        Map<Long, Set<Long>> bannerIdsByDualCheckAdGroupIds = StreamEx.of(activeAdGroups)
                .filter(adGroup -> dualModerationGroupTypes.contains(adGroup.getType()))
                .mapToEntry(AdGroup::getId, adGroup -> activeBannerIdsByActiveAdGroupIds.get(adGroup.getId()))
                .toMap();
        var pageBlocksAndBannerIds = getModeratedPagesAndBanners(shard, bannerIdsByDualCheckAdGroupIds);
        var pageBlocksByAdGroupIdFiltered = pageBlocksAndBannerIds.getLeft();
        // Добавляем пейджи для групп без двухэтапной модерации
        var nonDualCheckAdGroupIds = StreamEx.of(activeAdGroups)
                .map(AdGroup::getId)
                .remove(bannerIdsByDualCheckAdGroupIds::containsKey)
                .toSet();
        var nonDualCheckAdGroupsPageTargets = adGroupRepository.getAdGroupsPageTargetByAdGroupId(shard,
                nonDualCheckAdGroupIds);
        pageBlocksByAdGroupIdFiltered.putAll(nonDualCheckAdGroupsPageTargets);

        Map<Long, List<PageBlock>> excludedPageBlocksByAdGroupIdFiltered = new HashMap<>();
         // В имеющийся белый список PageID добавляем то, что из camp_options.allowed_page_ids
        var campaignIds = activeAdGroups.stream().map(AdGroup::getCampaignId).collect(toSet());
        var allowedDisallowedPageIdsByCampaignId = campaignRepository.getCampaignsAllowedAndDisallowedPageIdsMap(shard, campaignIds);

        // Если для кампании пустой список PageID - удаляем из списка
        var allowedPageIdsByCampaignIdCleared =
                EntryStream.of(allowedDisallowedPageIdsByCampaignId)
                        .mapValues(Pair::getLeft)
                        .filterValues(x -> !x.isEmpty())
                        .toMap();

        // Если для кампании пустой список PageID - удаляем из списка
        var disallowedPageIdsByCampaignIdCleared =
                EntryStream.of(allowedDisallowedPageIdsByCampaignId)
                        .mapValues(Pair::getRight)
                        .filterValues(x -> !x.isEmpty())
                        .toMap();

        activeAdGroups.forEach(adGroup -> {
            if (allowedPageIdsByCampaignIdCleared.containsKey(adGroup.getCampaignId())) {
                pageBlocksByAdGroupIdFiltered.computeIfAbsent(adGroup.getId(), e -> new ArrayList<>())
                        .addAll(mapList(allowedPageIdsByCampaignIdCleared.get(adGroup.getCampaignId()),
                                pageId -> new PageBlock().withPageId(pageId)));
            }
            if (disallowedPageIdsByCampaignIdCleared.containsKey(adGroup.getCampaignId())) {
                excludedPageBlocksByAdGroupIdFiltered.computeIfAbsent(adGroup.getId(), e -> new ArrayList<>())
                        .addAll(mapList(disallowedPageIdsByCampaignIdCleared.get(adGroup.getCampaignId()),
                                pageId -> new PageBlock().withPageId(pageId)));
            }
        });
        var dualModeratedBannerIds = pageBlocksAndBannerIds.getRight();
        // В выдаче оставляем либо те баннеры и группы, для которых нет двухэтапной модерации, либо те,
        // для которых модерация операторами вернула Yes
        var moderatedBannerIdsByActiveAdGroupIds = StreamEx.of(activeAdGroups)
                .mapToEntry(adGroup -> activeBannerIdsByActiveAdGroupIds.get(adGroup.getId()))
                .mapToValue((adGroup, banners) ->
                    banners.stream()
                            .filter(bannerId ->
                                    !dualModerationGroupTypes.contains(adGroup.getType())    // группа без двойной модерации либо
                                || dualModeratedBannerIds.contains(bannerId)     // баннер прошёл двойную модерацию)
                            )
                            .collect(Collectors.toSet()))
                .filterKeyValue((adGroup, banners) -> !banners.isEmpty()
                        || cidsToForceIncludeGroups.contains(adGroup.getCampaignId()))
                .mapKeys(AdGroup::getId)
                .toMap();

        return new AdGroupsFilteringResult(
                pageBlocksByAdGroupIdFiltered,
                excludedPageBlocksByAdGroupIdFiltered,
                moderatedBannerIdsByActiveAdGroupIds);
    }

    /**
     * Возвращает активные группы со списком активных баннеров.
     * Группы кампаний из cidsToForceIncludeGroups включаем, даже если нет активных баннеров.
     */
    @Nonnull
    public Map<Long, Set<Long>> getActiveAdGroupsWithActiveBanners(
            int shard, Collection<AdGroup> adGroups, boolean includeNotActive, Set<Long> cidsToForceIncludeGroups) {
        var adGroupIds = StreamEx.of(adGroups).map(AdGroup::getId).toList();

        // выбираем баннеры по группам и оставляем только активные баннеры и группы
        Map<Long, Set<Long>> bannerIdsByAdGroupIds = bannerRepository.type
                .getSafely(shard, bannerAdGroupIdFilter(adGroupIds), BannerWithSystemFields.class)
                .stream()
                .filter(banner -> banner.getStatusPostModerate() != BannerStatusPostModerate.REJECTED)
                .filter(banner -> (cidsToForceIncludeGroups.contains(banner.getCampaignId()) ||
                        includeNotActive || banner.getStatusModerate() == BannerStatusModerate.YES)
                        && banner.getStatusShow() && !banner.getStatusArchived())
                .collect(groupingBy(BannerWithSystemFields::getAdGroupId,
                        mapping(BannerWithSystemFields::getId, Collectors.toSet())));
        // оставляем только активные баннеры
        return StreamEx.of(adGroups)
                // выкидываем отклоненные
                .filter(adGroup -> adGroup.getStatusPostModerate() != StatusPostModerate.REJECTED)
                .mapToEntry(identity(), adGroup -> bannerIdsByAdGroupIds.getOrDefault(adGroup.getId(), emptySet()))
                .filterKeyValue((adGroup, banners) -> !banners.isEmpty()
                        || cidsToForceIncludeGroups.contains(adGroup.getCampaignId()))
                .mapKeys(AdGroup::getId)
                .toMap();
    }

    /**
     * Из переданной мапы ID групп и ID баннеров формирует мапу промодерированных пейджей по группам и
     * промодерированные баннеры.
     *
     * @param shard                 Шард
     * @param bannerIdsByAdGroupIds Мапа проверяемых ID групп и ID баннеров
     */
    @Nonnull
    public Pair<Map<Long, List<PageBlock>>, Set<Long>> getModeratedPagesAndBanners(
            int shard, Map<Long, Set<Long>> bannerIdsByAdGroupIds) {
        var pageTargetsByAdGroupId =
                adGroupRepository.getAdGroupsPageTargetByAdGroupId(shard, bannerIdsByAdGroupIds.keySet());
        var pageIdsByBannerIds = EntryStream.of(bannerIdsByAdGroupIds)
                .flatMapValues(StreamEx::of)
                .invert()   // adGroupIdsByBannerIds
                .mapValues(pageTargetsByAdGroupId::get) // PageBlocksByBannerIds
                .nonNullValues()
                .flatMapValues(StreamEx::of)
                .mapValues(PageBlock::getPageId)
                .grouping(toSet());
        var moderateBannerPagesByBannerIds =
                moderateBannerPagesRepository.getModerateBannerPagesByBannerId(shard, pageIdsByBannerIds);
        var moderatedPageIdsByBannerIds = EntryStream.of(moderateBannerPagesByBannerIds)
                .flatMapValues(StreamEx::of)
                .filterValues(page -> page.getStatusModerate() == StatusModerateBannerPage.YES)
                .mapValues(ModerateBannerPage::getPageId)
                .grouping(toSet());
        var moderatedPageIdsByAdGroupIds = EntryStream.of(bannerIdsByAdGroupIds)
                .flatMapValues(StreamEx::of)
                .mapValues(moderatedPageIdsByBannerIds::get)
                .nonNullValues()
                .flatMapValues(StreamEx::of)
                .grouping(toSet());
        var moderatedPagesByAdGroupIds = EntryStream.of(pageTargetsByAdGroupId)
                .filterKeys(moderatedPageIdsByAdGroupIds::containsKey)
                .flatMapValues(StreamEx::of)
                .filterKeyValue((adGroupId, pageBlock) -> moderatedPageIdsByAdGroupIds
                        .getOrDefault(adGroupId, emptySet())
                        .contains(pageBlock.getPageId()))
                .grouping();
        return Pair.of(moderatedPagesByAdGroupIds, moderatedPageIdsByBannerIds.keySet());
    }

    private Pair<List<Target>, Boolean> collectCampaignInfo(@Nullable FrontendData frontendData,
                                                            Long campaignId,
                                                            List<AdGroup> adGroupsForCampaign,
                                                            Map<Long, Creative> creativesByBannerId,
                                                            Map<Long, List<Long>> bannerIdsByAdGroupId,
                                                            Map<Long, List<RetargetingCondition>> retargetingConditionsByAdGroupId,
                                                            Map<Long, RetargetingCondition> brandSafetyRetargetingsByCampaignId,
                                                            Map<Long, List<PageBlock>> pageBlocksByAdGroupId,
                                                            Map<Long, List<PageBlock>> excludedPageBlocksByAdGroupId,
                                                            Map<Long, List<BidModifier>> bidModifiersByCampaignId,
                                                            Map<Long, CampaignForForecast> campaignByCampaignId,
                                                            Map<Long, MainPageTrafficType> mainPageTrafficTypeByCampaignId,
                                                            Map<Long, Set<FrontpageCampaignShowType>> frontpageTypesByCampaignId,
                                                            Map<Long, Goal> cryptaSegments,
                                                            Map<Long, TrafficTypeCorrections> trafficTypeCorrectionsByCampaignId,
                                                            Map<Long, Long> adGroupPriorities,
                                                            @Nullable PricePackage pricePackage) {
        if (adGroupsForCampaign.isEmpty()) {
            // Если группа новая, то передаем пустую группу
            if (frontendData != null && frontendData.getGroupId() == null) {
                adGroupsForCampaign = singletonList(new AdGroup());
            } else {
                return Pair.of(emptyList(), false);
            }
        }

        int adGroupsCount = adGroupsForCampaign.size();

        Map<Long, List<Long>> bannerIdsByAdGroupIdForCampaign = new HashMap<>(adGroupsCount);
        Map<Long, List<RetargetingCondition>> adGroupRetConditionsByAdGroupIdForCampaign = new HashMap<>(adGroupsCount);

        adGroupsForCampaign.forEach(adGroup -> {
            Long adGroupId = adGroup.getId();
            List<Long> bannerIds = bannerIdsByAdGroupId.get(adGroupId);
            if (bannerIds != null) {
                bannerIdsByAdGroupIdForCampaign.put(adGroupId, bannerIds);
            }

            var retargetingConditions = retargetingConditionsByAdGroupId.get(adGroupId);
            var bsRetargetingCondition = brandSafetyRetargetingsByCampaignId.get(campaignId);

            var retargetings = new ArrayList<RetargetingCondition>();
            if (retargetingConditions != null) {
                retargetings.addAll(retargetingConditions);
            }

            if (bsRetargetingCondition != null) {
                retargetings.add(bsRetargetingCondition);
            }

            if (!retargetings.isEmpty()) {
                adGroupRetConditionsByAdGroupIdForCampaign.put(adGroupId, retargetings);
            }
        });

        List<BidModifier> bidModifiersForCampaign = nvl(bidModifiersByCampaignId.get(campaignId), emptyList());
        Map<Long, BidModifierDesktop> bidModifierDesktopByAdGroupIdForCampaign = new HashMap<>(adGroupsCount);
        Map<Long, BidModifierMobile> bidModifierMobileByAdGroupIdForCampaign = new HashMap<>(adGroupsCount);
        Map<Long, BidModifierDemographics> bidModifierDemographicsByAdGroupIdForCampaign = new HashMap<>(adGroupsCount);

        var campaignParametersCorrections =
                ifNotNull(trafficTypeCorrectionsByCampaignId.get(campaignId), CampaignParametersCorrections::new);
        var frontpageTargetTags = convertFrontpageTypesToAdGroupTargetTags(frontpageTypesByCampaignId.get(campaignId));
        var frontpagePageGroupTags = convertFrontpageTypesToAdGroupPageGroupTags(frontpageTypesByCampaignId.get(campaignId));
        var mainPageTrafficType = mainPageTrafficTypeByCampaignId.get(campaignId);

        bidModifiersForCampaign.forEach(bidModifier -> {
            Long adGroupId = bidModifier.getAdGroupId();
            BidModifierType bidModifierType = bidModifier.getType();
            switch (bidModifierType) {
                case DESKTOP_MULTIPLIER:
                    bidModifierDesktopByAdGroupIdForCampaign.put(adGroupId, (BidModifierDesktop) bidModifier);
                    break;
                case MOBILE_MULTIPLIER:
                    bidModifierMobileByAdGroupIdForCampaign.put(adGroupId, (BidModifierMobile) bidModifier);
                    break;
                case DEMOGRAPHY_MULTIPLIER:
                    bidModifierDemographicsByAdGroupIdForCampaign.put(adGroupId, (BidModifierDemographics) bidModifier);
                    break;
            }
        });

        // Проверяем, нет ли групп на ключевых словах - прогнозатор не умеет давать по ним прогноз, так что
        // мы хотим сообщить пользователю, что не все группы участвовали в прогнозе. Соответственно
        // в ответе пользователю могут быть только ошибки, только результат, и результат + ошибки
        // (результат и сообщение, что не все группы учтены в результате)
        boolean containsKeywordGroup = StreamEx.of(adGroupsForCampaign).select(CpmBannerAdGroup.class)
                .anyMatch(ag -> ag.getCriterionType() == CriterionType.KEYWORD);

        List<Target> targets = StreamEx.of(adGroupsForCampaign)
                .map(adGroup -> {
                    Long adGroupId = adGroup.getId();
                    Boolean enableNonSkippableVideo = adGroup instanceof CpmVideoAdGroup
                            && ((CpmVideoAdGroup) adGroup).getIsNonSkippable();
                    var targetTags = getTargetTags(frontpageTargetTags, adGroup);
                    var orderTags = getOrderTags(frontpagePageGroupTags, adGroup);
                    var adGroupPriority = adGroupPriorities.get(adGroupId);
                    var projectParameters = convertProjectParamsConditionJsonToProjectParameters(
                            adGroup.getProjectParamConditions());

                    return targetFromAdGroup(frontendData,
                            adGroup,
                            defaultIfNull(campaignByCampaignId.get(campaignId), new Campaign()),
                            Optional.ofNullable(adGroupRetConditionsByAdGroupIdForCampaign.get(adGroupId)).orElse(emptyList()),
                            Optional.ofNullable(bannerIdsByAdGroupIdForCampaign.get(adGroupId)).orElse(emptyList()),
                            cryptaSegments,
                            bidModifierDesktopByAdGroupIdForCampaign.get(adGroupId),
                            bidModifierMobileByAdGroupIdForCampaign.get(adGroupId),
                            bidModifierDemographicsByAdGroupIdForCampaign.get(adGroupId),
                            campaignParametersCorrections,
                            mainPageTrafficType,
                            targetTags,
                            orderTags,
                            creativesByBannerId,
                            Optional.ofNullable(pageBlocksByAdGroupId.get(adGroupId)).orElse(emptyList()),
                            Optional.ofNullable(excludedPageBlocksByAdGroupId.get(adGroupId)).orElse(emptyList()),
                            enableNonSkippableVideo,
                            adGroupPriority == null ? null : AdGroupCpmPriceUtils.isDefaultPriority(adGroupPriority),
                            pricePackage,
                            projectParameters
                    );
                })
                .filter(Objects::nonNull)
                .toList();
        return Pair.of(targets, containsKeywordGroup);
    }

    /**
     * Собираем тэги из тэгов кампании и тэгов группы
     * Если кампания для главной, то из набора надо удалить устаревший "portal-trusted" и
     * добавить тэги специфичные для набора конкретных типов показа
     *
     * @param frontpageTags тэги, собранные на осове типов показа кампании для главной
     */
    private List<String> getTargetTags(ArrayList<String> frontpageTags, AdGroup adGroup) {
        frontpageTags.addAll(nvl(adGroup.getTargetTags(), emptyList()));
        var targetTags = frontpageTags.stream().distinct().collect(toList());

        // удаляем устаревший тэг "portal-trusted"
        targetTags.remove(TargetTagEnum.FRONTPAGE_TAG.getTypedValue());

        return targetTags;
    }

    /**
     * Собираем тэги из тэгов кампании и тэгов группы
     * Если кампания для главной, то из набора надо удалить устаревший "portal-trusted" и
     * добавить тэги специфичные для набора конкретных типов показа
     *
     * @param frontpageTags тэги, собранные на осове типов показа кампании для главной
     */
    private List<String> getOrderTags(ArrayList<String> frontpageTags, AdGroup adGroup) {
        frontpageTags.addAll(nvl(adGroup.getPageGroupTags(), emptyList()));
        var orderTags = frontpageTags.stream().distinct().collect(toList());

        // удаляем устаревший тэг "portal-trusted"
        orderTags.remove(PageGroupTagEnum.FRONTPAGE_TAG.getTypedValue());

        return orderTags;
    }

    /**
     * Собираем target tags для кампании на главной из типов показа
     */
    private ArrayList<String> convertFrontpageTypesToAdGroupTargetTags(
            @Nullable Set<FrontpageCampaignShowType> frontpageCampaignShowTypes) {
        if (frontpageCampaignShowTypes == null) {
            return new ArrayList<>();
        }

        var tags = new ArrayList<String>();
        for (var type : frontpageCampaignShowTypes) {
            switch (type) {
                case FRONTPAGE:
                    tags.add(TargetTagEnum.PORTAL_HOME_DESKTOP_TAG.getTypedValue());
                    break;
                case FRONTPAGE_MOBILE:
                    tags.add(TargetTagEnum.PORTAL_HOME_MOBILE_TAG.getTypedValue());
                    break;
                case BROWSER_NEW_TAB:
                    tags.addAll(List.of(
                            TargetTagEnum.PORTAL_HOME_NTP_TAG.getTypedValue(),
                            TargetTagEnum.PORTAL_HOME_NTP_CHROME_TAG.getTypedValue()));
                    break;
            }
        }

        return tags;
    }

    /**
     * Собираем page group tags для кампании на главной из типов показа
     */
    private ArrayList<String> convertFrontpageTypesToAdGroupPageGroupTags(
            @Nullable Set<FrontpageCampaignShowType> frontpageCampaignShowTypes) {
        if (frontpageCampaignShowTypes == null) {
            return new ArrayList<>();
        }

        var tags = new ArrayList<String>();
        for (var type : frontpageCampaignShowTypes) {
            switch (type) {
                case FRONTPAGE:
                    tags.add(PageGroupTagEnum.PORTAL_HOME_DESKTOP_TAG.getTypedValue());
                    break;
                case FRONTPAGE_MOBILE:
                    tags.add(PageGroupTagEnum.PORTAL_HOME_MOBILE_TAG.getTypedValue());
                    break;
                case BROWSER_NEW_TAB:
                    tags.addAll(List.of(
                            PageGroupTagEnum.PORTAL_HOME_NTP_TAG.getTypedValue(),
                            PageGroupTagEnum.PORTAL_HOME_NTP_CHROME_TAG.getTypedValue()));
                    break;
            }
        }

        return tags;
    }

    @Nullable
    private Target targetFromAdGroup(@Nullable FrontendData frontendData, AdGroup adGroup,
                                     CampaignForForecast campaign,
                                     List<RetargetingCondition> rtConditions,
                                     List<Long> adGroupBannerIds,
                                     Map<Long, Goal> goalIdToCryptaGoalMapping,
                                     @Nullable BidModifierDesktop bidModifierDesktop,
                                     @Nullable BidModifierMobile bidModifierMobile,
                                     @Nullable BidModifierDemographics bidModifierDemographics,
                                     @Nullable CampaignParametersCorrections campaignParametersCorrections,
                                     @Nullable MainPageTrafficType mainPageTrafficType,
                                     List<String> targetTags,
                                     List<String> orderTags,
                                     Map<Long, Creative> creativesByBannerId,
                                     List<PageBlock> allowedPageIds,
                                     List<PageBlock> disallowedPageIds,
                                     Boolean enableNonSkippableVideo,
                                     @Nullable Boolean isDefaultAdGroup,
                                     PricePackage pricePackage,
                                     @Nullable List<ProjectParameter> projectParameters) {
        AdGroupData data = AdGroupData.builder()
                .withFrontendData(frontendData)
                .withAdGroup(adGroup)
                .withCampaign(campaign)
                .withRetargetingConditions(rtConditions)
                .withBannerIds(adGroupBannerIds)
                .withGoalIdToCryptaGoalMapping(goalIdToCryptaGoalMapping)
                .withBidModifierDesktop(bidModifierDesktop)
                .withBidModifierMobile(bidModifierMobile)
                .withBidModifierDemographics(bidModifierDemographics)
                .withCorrections(campaignParametersCorrections)
                .withCreativesByBannerId(creativesByBannerId)
                .withPageBlocks(allowedPageIds)
                .withExcludedPageBlocks(disallowedPageIds)
                .withMainPageTrafficType(mainPageTrafficType)
                .withTargetTags(targetTags)
                .withOrderTags(orderTags)
                .withEnableNonSkippableVideo(enableNonSkippableVideo)
                .withIsDefaultAdGroup(isDefaultAdGroup)
                .withPricePackage(pricePackage)
                .withProjectParameters(projectParameters)
                .build();
        return new AdGroupDataConverter()
                .convertAdGroupDataToInventoriTarget(data);
    }

    private Map<Long, CampaignForForecast> getCampaignByCampaignId(int shard, @Nullable ClientId clientId,
                                                                   List<Long> campaignIds) {
        return campaignRepository.getCampaignsForForecastByCampaignIds(shard, clientId, campaignIds);
    }

    private Map<Long, Creative> getCreativesByBannerId(int shard, @Nullable ClientId clientId, List<Long> bannerIds) {
        List<BannerWithCreative> bannerCreatives =
                bannerRepository.type.getSafely(shard, bannerIds, BannerWithCreative.class);
        Map<Long, Long> bannerCreativesMap = StreamEx.of(bannerCreatives)
                .filter(bc -> nvl(bc.getCreativeStatusModerate(), BannerCreativeStatusModerate.NEW) !=
                    BannerCreativeStatusModerate.NO)
                .mapToEntry(BannerWithCreative::getId, BannerWithCreative::getCreativeId)
                .nonNullValues()
                .toMap();
        Collection<Long> creativeIds = bannerCreativesMap.values();
        List<Creative> creatives;
        if (clientId == null) {
            creatives = creativeRepository.getCreatives(shard, creativeIds);
        } else {
            creatives = creativeRepository.getCreatives(shard, clientId, creativeIds);
        }
        Map<Long, Creative> creativesById = StreamEx.of(creatives)
                .toMap(Creative::getId, identity());

        return EntryStream.of(bannerCreativesMap)
                .mapValues(creativesById::get)
                .filterValues(Objects::nonNull)
                .toMap();
    }

    public Map<Long, TrafficTypeCorrections> collectCampaignsCorrections(int shard, List<Long> campaignIds) {
        Map<Long, List<BidModifier>> bidModifiersByCampaignId = bidModifierRepository.getByCampaignIds(
                        shard,
                        campaignIds,
                        EnumSet.of(BidModifierType.BANNER_TYPE_MULTIPLIER,
                                BidModifierType.INVENTORY_MULTIPLIER),
                        EnumSet.of(BidModifierLevel.CAMPAIGN))
                .stream()
                .collect(Collectors.groupingBy(BidModifier::getCampaignId));
        Map<Long, TrafficTypeCorrections> result = null;
        result = campaignIds
                .stream()
                .distinct()
                .collect(HashMap::new, (map, cid) -> map.put(cid,
                        collectCampaignCorrection(nvl(bidModifiersByCampaignId.get(cid), emptyList()))), HashMap::putAll);
        return result;
    }

    private static TrafficTypeCorrections collectCampaignCorrection(List<BidModifier> bidModifiers) {
        Integer banner = null;
        Integer videoInpage = null;
        Integer videoInstream = null;
        Integer videoInApp = null;
        Integer videoInbanner = null;
        Integer videoRewarded = null;

        for (BidModifier bidModifier : bidModifiers) {
            if (bidModifier instanceof BidModifierInventory) {
                BidModifierInventory bidModifierInventory = (BidModifierInventory) bidModifier;
                for (BidModifierInventoryAdjustment adj : bidModifierInventory.getInventoryAdjustments()) {
                    switch (adj.getInventoryType()) {
                        case INSTREAM_WEB:
                        case PREROLL:
                        case MIDROLL:
                        case POSTROLL:
                        case PAUSEROLL:
                        case OVERLAY:
                        case POSTROLL_OVERLAY:
                        case POSTROLL_WRAPPER:
                        case INROLL:
                        case INROLL_OVERLAY:
                            videoInstream = adj.getPercent();
                            break;
                        case INPAGE:
                            videoInpage = adj.getPercent();
                            break;
                        case INAPP:
                        case INTERSTITIAL:
                        case FULLSCREEN:
                            videoInApp = adj.getPercent();
                            break;
                        case INBANNER:
                            videoInbanner = adj.getPercent();
                            break;
                        case REWARDED:
                            videoRewarded = adj.getPercent();
                    }
                }
            }
            if (bidModifier instanceof BidModifierBannerType) {
                BidModifierBannerType bidModifierBannerType = (BidModifierBannerType) bidModifier;
                for (BidModifierBannerTypeAdjustment adj : bidModifierBannerType.getBannerTypeAdjustments()) {
                    if (adj.getBannerType() == BannerType.CPM_BANNER) {
                        banner = adj.getPercent();
                    }
                }
            }
        }
        return new TrafficTypeCorrections(banner, videoInpage, videoInstream, videoInApp, videoInbanner,
                videoRewarded);
    }

    private Map<Long, MainPageTrafficType> collectMainPageTrafficType(
            @Nullable Map<Long, Set<FrontpageCampaignShowType>> frontpageTypesByCampaignId) {
        if (frontpageTypesByCampaignId == null) {
            return null;
        }

        return EntryStream.of(frontpageTypesByCampaignId)
                .mapValues(MainPageTrafficType::convert)
                .toMap();
    }

    private List<ProjectParameter> convertProjectParamsConditionJsonToProjectParameters(
            @Nullable List<Long> projectParamConditionIds) {
        if (projectParamConditionIds == null) {
            return  null;
        }
        if(projectParamConditionIds.isEmpty()) {
            return emptyList();
        }
        var paramConditions = projectParamConditionRepository
                .getProjectParamConditionsByConditionIds(projectParamConditionIds);
        var paramsData = paramConditions.stream()
                .map(ProjectParamCondition::getConjunctions)
                .flatMap(Collection::stream)
                .map(ProjectParamConjunction::getBbKeywords)
                .flatMap(Collection::stream)
                .collect(Collectors.groupingBy(
                    BbKeyword::getKeyword, Collectors.mapping(BbKeyword::getValue, Collectors.toList()))
                );
        return EntryStream.of(paramsData).mapKeyValue(ProjectParameter::new).toList();
    }

    public Set<Long> getPricePackageIdsForBooking(Collection<PricePackage> allPricePackages) {
        Set<String> targetTags = StreamEx.of(nvl(ppcPropertiesSupport.get(INVENTORI_TARGET_TAGS_NAME), "")
                        .split(","))
                .map(String::strip)
                .toSet();
        // Взяли все пакеты из них выбираем те кто Баннеры на Главной ИЛИ все таргет теги из списка targetTags (и нет лишних)
        return allPricePackages.stream()
                .filter(p -> p.isFrontpagePackage() ||
                        p.isFrontpageVideoPackage() ||
                        (p.getAllowedTargetTags() != null && targetTags != null &&
                                targetTags.containsAll(p.getAllowedTargetTags())))
                .map(PricePackage::getId)
                .collect(Collectors.toSet());
    }

    public class AdGroupsFilteringResult {
        private Map<Long, List<PageBlock>> pageBlocksByAdGroupIdFiltered;
        private Map<Long, List<PageBlock>> pageExcludedBlocksByAdGroupIdFiltered;
        private Map<Long, Set<Long>> moderatedBannerIdsByActiveAdGroupIds;

        private AdGroupsFilteringResult(Map<Long, List<PageBlock>> pageBlocksByAdGroupIdFiltered,
                                        Map<Long, List<PageBlock>> pageExcludedBlocksByAdGroupIdFiltered,
                                        Map<Long, Set<Long>> moderatedBannerIdsByActiveAdGroupIds) {
            this.pageBlocksByAdGroupIdFiltered = pageBlocksByAdGroupIdFiltered;
            this.pageExcludedBlocksByAdGroupIdFiltered = pageExcludedBlocksByAdGroupIdFiltered;
            this.moderatedBannerIdsByActiveAdGroupIds = moderatedBannerIdsByActiveAdGroupIds;
        }

        public Map<Long, List<PageBlock>> getPageBlocksByAdGroupIdFiltered() {
            return pageBlocksByAdGroupIdFiltered;
        }

        public Map<Long, Set<Long>> getModeratedBannerIdsByActiveAdGroupIds() {
            return moderatedBannerIdsByActiveAdGroupIds;
        }

        public Map<Long, List<PageBlock>> getPageExcludedBlocksByAdGroupIdFiltered() {
            return pageExcludedBlocksByAdGroupIdFiltered;
        }
    }
}
