package ru.yandex.direct.core.entity.inventori.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.currency.service.CurrencyRateService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.inventori.service.validation.InventoriDefectIds;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.inventori.model.request.BlockSize;
import ru.yandex.direct.inventori.model.request.CampaignParameters;
import ru.yandex.direct.inventori.model.request.CampaignParametersCorrections;
import ru.yandex.direct.inventori.model.request.CampaignParametersRf;
import ru.yandex.direct.inventori.model.request.CampaignParametersSchedule;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.inventori.model.request.InventoriCampaignType;
import ru.yandex.direct.inventori.model.request.StrategyType;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.request.TrafficTypeCorrections;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.calcPackagePrice;
import static ru.yandex.direct.feature.FeatureName.SOCIAL_ADVERTISING_BY_LAW;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для получения запросов в InventORI (CPM-прогнозатор).
 */
@ParametersAreNonnullByDefault
@Service
public class InventoriServiceCore {

    private static final int CHUNK_SIZE = 1000;
    /**
     * Все разрешенные форматы баннеров из
     * protected/Direct/Validation/Image.pm
     * <p>
     * Менять синхронно с perl-ом
     */
    public static final Set<BlockSize> ALLOWED_BLOCK_SIZES = ImmutableSet.of(
            new BlockSize(160, 600),
            new BlockSize(240, 400),
            new BlockSize(240, 600),
            new BlockSize(300, 250),
            new BlockSize(300, 300),
            new BlockSize(300, 500),
            new BlockSize(300, 600),
            new BlockSize(320, 50),
            new BlockSize(320, 100),
            new BlockSize(320, 480),
            new BlockSize(336, 280),
            new BlockSize(480, 320),
            new BlockSize(728, 90),
            new BlockSize(970, 250),
            new BlockSize(1000, 120)
    );

    public static final Set<BlockSize> ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_MOBILE = ImmutableSet.of(
            new BlockSize(2 * 320, 2 * 67)
    );

    public static final Set<BlockSize> ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_DESKTOP = ImmutableSet.of(
            new BlockSize(2 * 728, 2 * 90)
    );

    public static final Set<BlockSize> ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_BROWSER_NEW_TAB = ImmutableSet.of(
            new BlockSize(2 * 728, 2 * 90)
    );

    public static final Set<BlockSize> ALLOWED_BLOCK_SIZES_FOR_CPM_GEOPRODUCT = ImmutableSet.of(
            new BlockSize(640, 100)
    );

    public static final Set<BlockSize> ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_ALL_TYPES =
            Sets.union(Sets.union(ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_MOBILE,
                    ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_DESKTOP), ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_BROWSER_NEW_TAB);

    public static final Long DEFAULT_AVG_CPM_MICRO_RUB = 0L;

    private final CurrencyRateService currencyRateService;
    private final CampaignInfoCollector infoCollector;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageService pricePackageService;
    private final FeatureService featureService;

    @Autowired
    public InventoriServiceCore(
            CurrencyRateService currencyRateService,
            CampaignInfoCollector infoCollector,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            CampaignTypedRepository campaignTypedRepository,
            PricePackageService pricePackageService,
            FeatureService featureService) {
        this.currencyRateService = currencyRateService;
        this.infoCollector = infoCollector;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageService = pricePackageService;
        this.featureService = featureService;
    }

    /**
     * Возвращает CampaignPrediction запросы в InventORI.
     *
     * @param fromIntapi                 происходит ли вызов метода из intapi
     * @param operatorUid                uid оператора
     * @param clientId                   id клиента
     * @param campaignIds                id кампаний, для которых нужно сформировать запросы
     * @param includeNotActive           нужно ли учитывать группы с баннерами, которые не видны в БК, при
     *                                   формировании запросов
     * @param collectStrategyInfo        нужно ли добавлять в запросы информацию о стратегиях кампаний
     * @param collectCampaignCorrections нужно ли заполнять корректировки
     * @param fromBrandLiftEstimation                для Brand Lift нужен прогноз охвата для ручной стратегии, для таких кампаний
     *                                   заполняем дефолтные значения в CampaignParametersSchedule и смотрим только на
     *                                   target_reach
     * @return запросы в InventORI
     */
    public List<CampaignPredictionRequest> getCampaignPredictionRequestForCampaigns(
            boolean fromIntapi,
            @Nullable Long operatorUid,
            @Nullable ClientId clientId,
            List<Long> campaignIds,
            boolean includeNotActive,
            boolean collectStrategyInfo,
            boolean collectCampaignCorrections,
            boolean fromBrandLiftEstimation,
            ValidationResult<List<Long>, Defect> validationResult
    ) {
        List<CampaignPredictionRequest> response = new ArrayList<>(campaignIds.size());

        shardHelper.groupByShard(campaignIds, ShardKey.CID).chunkedBy(CHUNK_SIZE).forEach((shard, cids) -> {
            Map<Long, TrafficTypeCorrections> trafficTypeCorrectionsByCampaignId =
                    collectCampaignCorrections ? infoCollector.collectCampaignsCorrections(shard, cids) : emptyMap();

            Map<Long, Pair<List<Target>, Boolean>> campaignsInfo =
                    infoCollector.collectCampaignsInfoWithClientIdAndUid(null, fromIntapi, shard, operatorUid,
                            clientId, cids,
                            includeNotActive,
                            trafficTypeCorrectionsByCampaignId);

            List<Campaign> campaigns = campaignRepository.getCampaigns(shard, cids);
            Set<ClientId> clientIds = campaigns.stream()
                    .map(Campaign::getClientId)
                    .map(ClientId::fromLong)
                    .collect(toSet());
            Map<ClientId, Boolean> isSocialByLawByClientId =
                    featureService.isEnabledForClientIdsOnlyFromDb(clientIds, SOCIAL_ADVERTISING_BY_LAW.getName());

            var campaignTypeMap = listToMap(campaigns, Campaign::getId, Campaign::getType);

            Map<Long, CampaignPredictionRequest.Builder> builderByCampaignId = StreamEx.of(campaignsInfo.entrySet())
                        .mapToEntry(Map.Entry::getKey, e -> CampaignPredictionRequest.builder()
                                .withCampaignId(e.getKey())
                                .withCampaignType(getInventoriCampaignType(campaignTypeMap.get(e.getKey()), e.getValue().getLeft()))
                                .withTargets(e.getValue().getLeft())
                                .containsKeywordGroup(e.getValue().getRight()))
                        .toMap();
                if (collectStrategyInfo) {
                    var brandLifts = campaignRepository.getBrandSurveyIdsForCampaigns(shard, cids);
                    Map<Long, PricePackage> pricePackages =
                            pricePackageService.getPricePackageByCampaigns(shard, campaigns);

                    Map<Long, String> problemCamps = new HashMap<>();
                    campaigns.forEach(campaign -> {
                        Long campaignId = campaign.getId();
                        try {
                            CampaignPredictionRequest.Builder builder = builderByCampaignId.get(campaignId);
                            if (builder == null) {
                                return;
                            }
                            builder.withIsSocial(isSocialByLawByClientId.get(ClientId.fromLong(campaign.getClientId())));

                            DbStrategy strategy = campaign.getStrategy();

                            // костыль для ручной стратегии, используется исключительно для получения target_reach
                            // выпилить после INVENTORI-3728
                            if (fromBrandLiftEstimation && strategy.getStrategyName() == StrategyName.CPM_DEFAULT) {
                                var strategyData = new StrategyData()
                                        .withBudget(BigDecimal.valueOf(1))
                                        .withAvgCpm(BigDecimal.valueOf(10))
                                        .withAutoProlongation(1L)
                                        .withStart(LocalDate.now())
                                        .withFinish(LocalDate.now().plusDays(1));
                                strategy.setStrategyData(strategyData);
                                strategy.setStrategyName(StrategyName.AUTOBUDGET_MAX_IMPRESSIONS);
                                strategy.setRf(0);
                            }

                            BigDecimal budget = strategy.getStrategyData().getBudget();
                            if (budget == null) {
                                budget = strategy.getStrategyData().getSum();
                            }

                            CurrencyCode clientCurrency = campaign.getCurrency();

                            Long autoProlongation = strategy.getStrategyData().getAutoProlongation();
                            BigDecimal avgCpm = strategy.getStrategyData().getAvgCpm();
                            BigDecimal avgCpv = strategy.getStrategyData().getAvgCpv();
                            PricePackage pricePackage = pricePackages.get(campaign.getId());

                            StrategyType strategyType = getStrategyType(strategy.getStrategyName());
                            CampaignParametersSchedule schedule = CampaignParametersSchedule.builder()
                                    .withStrategyType(strategyType)
                                    .withBudget(convertFromClientsCurrency(budget, clientCurrency))
                                    .withStartDate(nvl(strategy.getStrategyData().getStart(), LocalDate.now()))
                                    .withEndDate(
                                            nvl(strategy.getStrategyData().getFinish(), LocalDate.now().plusDays(6)))
                                    .withCpm(calcAvgCpm(
                                            strategyType, avgCpm, clientCurrency, pricePackage, campaign.getId()))
                                    .withCpv(calcAvgCpv(avgCpv, clientCurrency))
                                    .withAutoProlongation(autoProlongation == null ? null : autoProlongation == 1)
                                    .withIsBooking(pricePackage == null ? null : pricePackage.isFrontpagePackage())
                                    .build();
                            CampaignParametersRf rf = new CampaignParametersRf(strategy.getRf(), getRfReset(strategy));

                            CampaignParameters campaignParameters = CampaignParameters.builder()
                                    .withSchedule(schedule)
                                    .withRf(rf)
                                    .withCorrections(ifNotNull(
                                            trafficTypeCorrectionsByCampaignId.get(campaignId),
                                            CampaignParametersCorrections::new))
                                    .build();

                            builder.withParameters(campaignParameters);

                            builder.withBrandlift(brandLifts.get(campaignId) != null);

                            // Для баннерки на главной позволяем пребукинг (когда еще нет креативов, но кампания уже
                            // бронирует показы в инвентори), там не отсееваем баннеры без креативов
                            if (pricePackage == null ||
                                    !(pricePackage.isFrontpagePackage() || pricePackage.isFrontpageVideoPackage())) {
                                builder.withTargets(
                                        StreamEx.of(campaignsInfo.get(campaignId).getLeft())
                                                .filter(Target::hasCreatives)
                                                .toList());
                            }

                        } catch (Exception e) {
                            problemCamps.put(campaignId, e.toString());
                        }
                    });
                    ListValidationBuilder<Long, Defect> vb = new ListValidationBuilder<>(validationResult);
                    vb.checkEach(campaignsHaveUnexpectedProblemWithCampaign(problemCamps));
                }
                response.addAll(mapList(builderByCampaignId.values(), CampaignPredictionRequest.Builder::build));
        });

        return response;
    }

    public static InventoriCampaignType getInventoriCampaignType(@Nullable CampaignType campaignType, @Nullable List<Target> targets) {
        if (CampaignType.CPM_PRICE.equals(campaignType)) {
            return InventoriCampaignType.FIX_CPM;
        }
        if (CampaignType.CPM_YNDX_FRONTPAGE.equals(campaignType) ||
                targets != null && targets.stream().anyMatch(t -> GroupType.MAIN_PAGE_AND_NTP.equals(t.getGroupType()))) {
            return InventoriCampaignType.MAIN_PAGE_AND_NTP;
        }

        return InventoriCampaignType.MEDIA_RSYA;
    }

    private Constraint<Long, Defect> campaignsHaveUnexpectedProblemWithCampaign(Map<Long, String> problemCamps) {
        return campaignId -> {
            if (problemCamps.containsKey(campaignId)) {
                return new Defect<>(InventoriDefectIds.String.UNEXPECTED_INVENTORI_PROBLEM_WITH_CAMPAIGN,
                        new StringDefectParams().withInvalidSubstrings(
                                singletonList(problemCamps.get(campaignId))));
            }
            return null;
        };
    }

    public long convertFromClientsCurrency(double value, CurrencyCode clientCurrency) {
        return currencyRateService.convertMoney(Money.valueOf(value, clientCurrency), CurrencyCode.RUB).micros();
    }

    private long convertFromClientsCurrency(BigDecimal value, CurrencyCode clientCurrency) {
        return currencyRateService.convertMoney(Money.valueOf(value, clientCurrency), CurrencyCode.RUB).micros();
    }

    public Long calcAvgCpm(StrategyType strategyType, Number avgCpm,
                           CurrencyCode campCurrency, @Nullable PricePackage pricePackage, @Nullable Long campaignId) {

        if (strategyType == StrategyType.MAX_AVG_CPV || strategyType == StrategyType.MAX_AVG_CPV_CUSTOM_PERIOD) {
            return null;
        }
        // если стоимость указана в стратегии, используем ее.
        if (avgCpm != null) {
            if (avgCpm instanceof BigDecimal) {
                return convertFromClientsCurrency((BigDecimal) avgCpm, campCurrency);
            }
            return convertFromClientsCurrency(avgCpm.doubleValue(), campCurrency);
        }
        // если есть пакет, то используем цену оттуда
        if (pricePackage != null) {
            if (campaignId != null) {
                Map<Long, CampaignWithPricePackage> cpmPriceCampaigns = campaignTypedRepository.getTypedCampaignsMap(
                        shardHelper.getShardByCampaignId(campaignId),
                        Map.of(campaignId, CampaignType.CPM_PRICE), CampaignType.CPM_PRICE);
                var campaign = cpmPriceCampaigns.get(campaignId);
                if (campaign != null) {
                    boolean backendCpmPriceCampaignBudgetCalcEnabled = featureService.isEnabledForClientId(
                            ClientId.fromLong(campaign.getClientId()),
                            FeatureName.BACKEND_CPM_PRICE_CAMPAIGN_BUDGET_CALC_ENABLED);
                    BigDecimal totalPrice = calcPackagePrice(campaign, pricePackage, false,
                            backendCpmPriceCampaignBudgetCalcEnabled);
                    return convertFromClientsCurrency(totalPrice, pricePackage.getCurrency());
                }
            }
            return convertFromClientsCurrency(pricePackage.getPrice(), pricePackage.getCurrency());
        }
        // иначе дефолтный 0L
        return DEFAULT_AVG_CPM_MICRO_RUB;
    }

    public Long calcAvgCpv(BigDecimal avgCpv, CurrencyCode campCurrency) {
        return ifNotNull(avgCpv, cpv -> convertFromClientsCurrency(cpv, campCurrency));
    }

    /**
     * Возвращает тип автобюджетной стратегии (MAX_REACH или MIN_CPM) по ее названию.
     *
     * @param strategyName название стратегии
     * @return тип стратегии
     */
    private static StrategyType getStrategyType(StrategyName strategyName) {
        switch (strategyName) {
            case AUTOBUDGET_MAX_REACH:
                return StrategyType.MAX_REACH;
            case AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD:
                return StrategyType.MAX_REACH_CUSTOM_PERIOD;
            case AUTOBUDGET_MAX_IMPRESSIONS:
                return StrategyType.MIN_CPM;
            case AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD:
                return StrategyType.MIN_CPM_CUSTOM_PERIOD;
            case AUTOBUDGET_AVG_CPV:
                return StrategyType.MAX_AVG_CPV;
            case AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD:
                return StrategyType.MAX_AVG_CPV_CUSTOM_PERIOD;
            case PERIOD_FIX_BID:
                return StrategyType.FIX_PRICE;
        }
        throw new IllegalArgumentException();
    }

    /**
     * Получить параметр rfReset из стратегии. Если rf = 0, то возвращаем 0. Сделано из-за того, что при отключении
     * ограничения показов контроллер saveCamp сохраняет 0 только в rf, а rfReset остается прежним.
     *
     * @param dbStrategy стратегия кампании
     * @return значение rfReset
     */
    public Integer getRfReset(DbStrategy dbStrategy) {
        if (dbStrategy.getRf() != null && dbStrategy.getRf() == 0) {
            return 0;
        }

        return dbStrategy.getRfReset();
    }

    public List<Long> getBookedYndxFrontpageCampaigns() {
        Set<Long> pricePackageIds = pricePackageService.getActivePricePackages(
                Set.of(AdGroupType.CPM_YNDX_FRONTPAGE, AdGroupType.CPM_VIDEO))
                .entrySet().stream()
                .filter(it -> it.getValue().isFrontpagePackage() || it.getValue().isFrontpageVideoPackage())
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
        List<Long> campaignIds = shardHelper.dbShards().stream()
                .map(shard -> campaignRepository.getBookedCampaignIds(shard, pricePackageIds))
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
        return campaignIds;
    }

    public List<Long> getBookedCampaigns() {
        Map<Long, PricePackage> pricePackages = pricePackageService.getActivePricePackages(null);
        // Взяли все пакеты из них выбираем те кто Баннеры на Главной ИЛИ все таргет теги из списка targetTags (и нет лишних)
        Set<Long> pricePackageIds = infoCollector.getPricePackageIdsForBooking(pricePackages.values());

        List<Long> campaignIds = shardHelper.dbShards().stream()
                .map(shard -> campaignRepository.getBookedCampaignIds(shard, pricePackageIds))
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
        return campaignIds;
    }
}
