package ru.yandex.direct.core.entity.inventori.service.type;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSize;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.campaign.model.CampaignForForecast;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.inventori.model.request.AudienceGroup;
import ru.yandex.direct.inventori.model.request.AudioCreative;
import ru.yandex.direct.inventori.model.request.BlockSize;
import ru.yandex.direct.inventori.model.request.CryptaGroup;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.inventori.model.request.MainPageTrafficType;
import ru.yandex.direct.inventori.model.request.MobileOsType;
import ru.yandex.direct.inventori.model.request.PageBlock;
import ru.yandex.direct.inventori.model.request.PlatformCorrections;
import ru.yandex.direct.inventori.model.request.ProfileCorrection;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.request.VideoCreative;
import ru.yandex.direct.utils.CollectionUtils;

import static com.google.common.math.IntMath.gcd;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.CPM_PRICE_PREMIUM_FORMAT_CREATIVE_SIZE;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.CPM_PRICE_PREMIUM_FORMAT_MOBILE_CREATIVE_SIZE;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.DESKTOP_DEFAULT_CREATIVE_SIZE;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.MOBILE_DEFAULT_CREATIVE_SIZE;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES_FOR_CPM_GEOPRODUCT;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_ALL_TYPES;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_BROWSER_NEW_TAB;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_DESKTOP;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_MOBILE;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.AUDIENCE_GROUPS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.AUDIO_CREATIVES;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.BLOCK_SIZES;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.CORRECTIONS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.CRYPTA_GROUPS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.DOMAINS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.ENABLE_NON_SKIPPABLE_VIDEO;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.EXCLUDED_BRAND_SAFETY_CATEGORIES;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.EXCLUDED_DOMAINS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.EXCLUDED_PAGE_BLOCKS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.EXCLUDED_SSP;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.GENRES_AND_CATEGORIES;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.IS_DEFAULT_GROUP;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.MAIN_PAGE_TRAFFIC_TYPE;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.ORDER_TAGS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.PAGE_BLOCKS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.PLATFORM_CORRECTIONS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.PROFILE_CORRECTIONS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.PROJECT_PARAMETERS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.REGIONS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.SSP;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.TARGET_TAGS;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.Field.VIDEO_CREATIVES;
import static ru.yandex.direct.core.entity.inventori.service.type.CommonConverters.convertCreativeWithAdditionalData;
import static ru.yandex.direct.core.entity.inventori.service.type.CommonConverters.convertGroupType;
import static ru.yandex.direct.core.entity.inventori.service.type.CommonConverters.convertPageBlocks;
import static ru.yandex.direct.core.entity.inventori.service.type.CommonConverters.convertRuleTypeToAudienceGroupingType;
import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.all;
import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.long_term;
import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.short_term;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.AUDIENCE;
import static ru.yandex.direct.inventori.model.request.GroupType.AUDIO;
import static ru.yandex.direct.inventori.model.request.GroupType.BANNER;
import static ru.yandex.direct.inventori.model.request.GroupType.BANNER_IN_GEO_APPS;
import static ru.yandex.direct.inventori.model.request.GroupType.BANNER_IN_METRO;
import static ru.yandex.direct.inventori.model.request.GroupType.GEOPRODUCT;
import static ru.yandex.direct.inventori.model.request.GroupType.GEO_PIN;
import static ru.yandex.direct.inventori.model.request.GroupType.INDOOR;
import static ru.yandex.direct.inventori.model.request.GroupType.MAIN_PAGE_AND_NTP;
import static ru.yandex.direct.inventori.model.request.GroupType.OUTDOOR;
import static ru.yandex.direct.inventori.model.request.GroupType.VIDEO;
import static ru.yandex.direct.utils.CollectionUtils.defaultIfEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертер данных группы {@link AdGroupData} в таргетинг inventori {@link Target}.
 */
//todo почистить методы внизу в рамках DIRECT-104384
public class AdGroupDataConverter {
    public static final BlockSize VIDEO_PROPORTION_16_9 = new BlockSize(16, 9);

    private static final String UNDEFINED_KEYWORD = "0";
    private static final String INCOME_B1 = "618:1";
    public static final String INCOME_B2 = "618:2";

    private static final Map<GroupType, Set<Field>> FIELDS_BY_GROUP_TYPES =
            ImmutableMap.<GroupType, Set<Field>>builder()
                    .put(BANNER,
                            Set.of(REGIONS,
                                    DOMAINS,
                                    EXCLUDED_DOMAINS,
                                    SSP,
                                    EXCLUDED_SSP,
                                    BLOCK_SIZES,
                                    CRYPTA_GROUPS,
                                    AUDIENCE_GROUPS,
                                    GENRES_AND_CATEGORIES,
                                    EXCLUDED_BRAND_SAFETY_CATEGORIES,
                                    PLATFORM_CORRECTIONS,
                                    CORRECTIONS,
                                    PAGE_BLOCKS,
                                    EXCLUDED_PAGE_BLOCKS,
                                    TARGET_TAGS,
                                    ORDER_TAGS,
                                    PROJECT_PARAMETERS))
                    .put(VIDEO,
                            Set.of(REGIONS,
                                    DOMAINS,
                                    EXCLUDED_DOMAINS,
                                    SSP,
                                    EXCLUDED_SSP,
                                    VIDEO_CREATIVES,
                                    CRYPTA_GROUPS,
                                    AUDIENCE_GROUPS,
                                    GENRES_AND_CATEGORIES,
                                    EXCLUDED_BRAND_SAFETY_CATEGORIES,
                                    PLATFORM_CORRECTIONS,
                                    CORRECTIONS,
                                    ENABLE_NON_SKIPPABLE_VIDEO,
                                    IS_DEFAULT_GROUP,
                                    PAGE_BLOCKS,
                                    EXCLUDED_PAGE_BLOCKS,
                                    TARGET_TAGS,
                                    ORDER_TAGS,
                                    PROJECT_PARAMETERS))
                    .put(AUDIO,
                            Set.of(REGIONS,
                                    AUDIO_CREATIVES,
                                    CRYPTA_GROUPS,
                                    AUDIENCE_GROUPS,
                                    GENRES_AND_CATEGORIES,
                                    EXCLUDED_BRAND_SAFETY_CATEGORIES,
                                    PLATFORM_CORRECTIONS,
                                    CORRECTIONS))
                    .put(INDOOR,
                            Set.of(PAGE_BLOCKS,
                                    VIDEO_CREATIVES,
                                    EXCLUDED_BRAND_SAFETY_CATEGORIES,
                                    PROFILE_CORRECTIONS))
                    .put(OUTDOOR,
                            Set.of(PAGE_BLOCKS,
                                    EXCLUDED_BRAND_SAFETY_CATEGORIES,
                                    VIDEO_CREATIVES))
                    .put(MAIN_PAGE_AND_NTP,
                            Set.of(REGIONS,
                                    BLOCK_SIZES,
                                    MAIN_PAGE_TRAFFIC_TYPE,
                                    PLATFORM_CORRECTIONS,
                                    IS_DEFAULT_GROUP,
                                    TARGET_TAGS,
                                    ORDER_TAGS))
                    .put(BANNER_IN_GEO_APPS,
                            Set.of(REGIONS,
                                    BLOCK_SIZES,
                                    CRYPTA_GROUPS,
                                    AUDIENCE_GROUPS,
                                    PLATFORM_CORRECTIONS,
                                    TARGET_TAGS,
                                    ORDER_TAGS))
                    .put(GEO_PIN,
                            Set.of(REGIONS,
                                    CRYPTA_GROUPS,
                                    AUDIENCE_GROUPS,
                                    PLATFORM_CORRECTIONS))
                    .build();

    private static final Set<GoalType> GENRES_AND_CATEGORIES_GOAL_TYPES =
            Set.of(GoalType.AUDIO_GENRES, GoalType.CONTENT_CATEGORY, GoalType.CONTENT_GENRE);
    private static final Set<GoalType> BRAND_SAFETY_GOAL_TYPES = Set.of(GoalType.BRANDSAFETY);
    private static final String BRAND_SAFETY_KEYWORD_TYPE = "931";
    private static final String BRAND_SAFETY_KEYWORD_NAME = "brand-safety-categories";

    /**
     * Конвертация данных группы в таргетинг inventori.
     * Возвращает null, в случае, если группу не надо включать в таргетинг.
     *
     * @param data Данные группы
     * @return Таргетинг inventori.
     */
    @Nullable
    public Target convertAdGroupDataToInventoriTarget(AdGroupData data) {
        GroupType groupType = getGroupType(data);

        // На группы с ключевыми слова прогноз не считаем
        if (data.getAdGroup() instanceof CpmBannerAdGroup
                && CriterionType.KEYWORD.equals(((CpmBannerAdGroup) data.getAdGroup()).getCriterionType())) {
            return null;
        }

        Target target = new Target()
                .withGroupType(groupType)
                .withAdGroupId(ifNotNull(data.getAdGroup(), AdGroup::getId));

        addRegions(target, groupType, data);
        addDomains(target, groupType, data);
        addExcludedDomains(target, groupType, data);
        addSsp(target, groupType, data);
        addExcludedSsp(target, groupType, data);
        addBlockSizes(target, groupType, data);
        addVideoCreatives(target, groupType, data);
        addAudioCreatives(target, groupType, data);
        addCryptaGroups(target, groupType, data);
        addAudienceGroups(target, groupType, data);
        addGenresAndCategories(target, groupType, data);
        addExcludedBrandSafetyCategories(target, groupType, data);
        addPageBlocks(target, groupType, data);
        addExcludedPageBlocks(target, groupType, data);
        addMainPageTrafficType(target, groupType, data);
        addPlatformCorrections(target, groupType, data);
        addProfileCorrections(target, groupType, data);
        addCorrections(target, groupType, data);
        addTargetTags(target, groupType, data);
        addOrderTags(target, groupType, data);
        addEnableNonSkipableVideo(target, groupType, data);
        addIsDefaultGroup(target, groupType, data);
        addProjectParameters(target, groupType, data);

        return target;
    }

    private GroupType getGroupType(AdGroupData data) {
        if (data.getAdGroup().getType() != null) {
            return convertGroupType(data.getAdGroup().getType());
        } else if (data.getMainPageTrafficType() != null) {
            // Для медийки на главной фронт передает тип нацеливания BANNER вместо MAIN_PAGE_AND_NTP, поэтому нужно
            // исправить значение, если кампания имеет тип траффика главной
            return MAIN_PAGE_AND_NTP;
        } else {
            GroupType groupType = ifNotNull(data.getFrontendData(), FrontendData::getGroupType);

            if (groupType == GroupType.VIDEO_NON_SKIPPABLE) {
                return VIDEO;
            }

            // GEOPRODUCT и BANNER_IN_METRO - это алиасы для BANNER_IN_GEO_APPS (DIRECT-103936, DIRECT-112325),
            // но Inventori хочет именно BANNER_IN_GEO_APPS.
            return (groupType == GEOPRODUCT || groupType == BANNER_IN_METRO) ? BANNER_IN_GEO_APPS : groupType;
        }
    }

    private GroupType getCorrectGroupType(GroupType groupType) {

        if (groupType == GroupType.VIDEO_NON_SKIPPABLE) {
                return VIDEO;
        }

        // GEOPRODUCT и BANNER_IN_METRO - это алиасы для BANNER_IN_GEO_APPS (DIRECT-103936, DIRECT-112325),
        // но Inventori хочет именно BANNER_IN_GEO_APPS.
        return (groupType == GEOPRODUCT || groupType == BANNER_IN_METRO) ? BANNER_IN_GEO_APPS : groupType;
    }

    private void addRegions(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(REGIONS)) {
            Set<Integer> regions = ifNotNull(data.getFrontendData(), FrontendData::getGeo);
            regions = nvl(regions, getRegions(data.getCampaign().getGeo(), data.getAdGroup().getGeo()));
            target.withRegions(regions);
        }
    }

    private void addTargetTags(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(TARGET_TAGS)) {
            List<String> frontendTargetTags = ifNotNull(data.getFrontendData(), FrontendData::getTargetTags);
            List<String> dbTargetTags = data.getTargetTags();
            target.withTargetTags(nullableNvl(frontendTargetTags, dbTargetTags));
        }
    }

    private void addOrderTags(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(ORDER_TAGS)) {
            List<String> dbOrderTags = data.getOrderTags();
            target.withOrderTags(dbOrderTags);
        }
    }

    private void addEnableNonSkipableVideo(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(ENABLE_NON_SKIPPABLE_VIDEO)) {
            target.withEnableNonSkippableVideo(data.getEnableNonSkippableVideo());
        }
    }

    private void addIsDefaultGroup(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(IS_DEFAULT_GROUP) && data.getIsDefaultAdGroup() != null) {
            target.withIsDefaultGroup(data.getIsDefaultAdGroup());
        }
    }

    private void addExcludedDomains(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(EXCLUDED_DOMAINS)) {
            CampaignForForecast campaign = data.getCampaign();
            Set<String> disabledVideoPlacements = new HashSet<>(nvl(campaign.getDisabledVideoPlacements(), emptySet()));
            Set<String> disabledDomains = nvl(campaign.getDisabledDomains(), emptySet());

            Set<String> excludedDomains = groupType == VIDEO ? disabledVideoPlacements : disabledDomains;
            target.withExcludedDomains(defaultIfEmpty(excludedDomains, null));
        }
    }

    private void addDomains(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(DOMAINS)) {
            CampaignForForecast campaign = data.getCampaign();
            Set<String> allowedDomains = new HashSet<>(nvl(campaign.getAllowedDomains(), emptySet()));
            target.withDomains(defaultIfEmpty(allowedDomains, null));
        }
    }

    private void addSsp(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(SSP)) {
            CampaignForForecast campaign = data.getCampaign();
            Set<String> allowedSsp = new HashSet<>(nvl(campaign.getAllowedSsp(), emptySet()));
            target.withSsp(defaultIfEmpty(allowedSsp, null));
        }
    }

    private void addExcludedSsp(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(EXCLUDED_SSP)) {
            CampaignForForecast campaign = data.getCampaign();
            Set<String> excludedSsp = new HashSet<>(nvl(campaign.getDisabledSsp(), emptySet()));
            target.withExcludedSsp(defaultIfEmpty(excludedSsp, null));
        }
    }

    private void addBlockSizes(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(BLOCK_SIZES) && isCpmType(data.getCampaign().getType())) {
            List<BlockSize> allAllowedBlockSizes = new ArrayList<>(getAllowedBlockSizes(groupType,
                    data.getMainPageTrafficType()));
            List<BlockSize> blockSizes = null;

            //Данные фронта
            if (data.getFrontendData() != null) {
                if (data.getFrontendData().getHasAdaptiveCreative() == Boolean.TRUE) {
                    blockSizes = allAllowedBlockSizes;
                } else {
                    blockSizes = ifNotNull(data.getFrontendData().getBlockSizes(),
                            frontBlockSizes -> mapList(frontBlockSizes, bs -> new BlockSize(bs.getWidth(),
                                    bs.getHeight())));
                }
            }

            //Данные базы
            if (blockSizes == null) {
                List<Creative> creatives = StreamEx.of(data.getBannerIds())
                        .map(data.getCreativesByBannerId()::get)
                        .filter(Objects::nonNull)
                        .filter(c -> c.getStatusModerate() != StatusModerate.NO)
                        .toList();

                boolean hasAdaptiveCreative = StreamEx.of(creatives)
                        .anyMatch(Creative::getIsAdaptive);

                // Если среди креативов есть адаптивный, нужно учитывать все поддерживаемые размеры блоков
                if (hasAdaptiveCreative) {
                    blockSizes = allAllowedBlockSizes;
                } else {
                    blockSizes = StreamEx.of(creatives)
                            .map(this::getBlockSizeByCreative)
                            .filter(Objects::nonNull)
                            .toList();
                }
                // DIRECT-161737: Передавать дефолтные размеры креативов для холодных буков
                // То есть для прайсовых кампаний, для баннеров на главной, которые надо показывать,
                // в случае отсутствия баннеров (еще не созданы), отправлять размеры потенциальных баннеров
                if (data.getPricePackage() != null
                        && nvl(blockSizes, emptyList()).isEmpty()
                        && data.getMainPageTrafficType() != null) {
                    Boolean isAllowPremiumDesktopCreative = data.getPricePackage()
                            .getTargetingsFixed().getAllowPremiumDesktopCreative();
                    switch (data.getMainPageTrafficType()) {
                        case DESKTOP:
                        case BROWSER_NEW_TAB:
                            blockSizes = (isAllowPremiumDesktopCreative) ?
                                    List.of(getBlockSizeByCreativeSize(CPM_PRICE_PREMIUM_FORMAT_CREATIVE_SIZE)) :
                                    List.of(getBlockSizeByCreativeSize(DESKTOP_DEFAULT_CREATIVE_SIZE));
                            break;
                        case MOBILE:
                            blockSizes = (isAllowPremiumDesktopCreative) ?
                                    mapList(CPM_PRICE_PREMIUM_FORMAT_MOBILE_CREATIVE_SIZE, this::getBlockSizeByCreativeSize) :
                                    List.of(getBlockSizeByCreativeSize(MOBILE_DEFAULT_CREATIVE_SIZE));
                            break;
                    }
                }
            }

            target.withBlockSizes(blockSizes);
        }
    }

    public static boolean allowedAddBlockSizes(Target target) {
        return FIELDS_BY_GROUP_TYPES.get(target.getGroupType()).contains(BLOCK_SIZES);
    }

    //todo Возможно стоит всегда в видео креативе отправлять duration, пропорции и абсолютные размеры
    //решить в рамках DIRECT-104384
    private void addVideoCreatives(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(VIDEO_CREATIVES)) {
            List<VideoCreative> videoCreatives;
            if (groupType == INDOOR) {
                videoCreatives = ifNotNull(data.getFrontendData(), frontendData ->
                        ifNotNull(frontendData.getCreatives(),
                                creatives -> mapList(creatives, this::convertVideoCreative)));
                videoCreatives = nvl(videoCreatives,
                        getIndoorVideoCreativesByBanners(data.getBannerIds(), data.getCreativesByBannerId()));
            } else if (groupType == OUTDOOR) {
                videoCreatives = ifNotNull(data.getFrontendData(), frontendData ->
                        ifNotNull(frontendData.getCreatives(),
                                creatives -> mapList(creatives, this::getVideoCreativeWithAdditionalData)));
                videoCreatives = nvl(videoCreatives,
                        getOutdoorVideoCreativesByBanners(data.getBannerIds(), data.getCreativesByBannerId()));
            } else {
                videoCreatives = ifNotNull(data.getFrontendData(), FrontendData::getVideoCreatives);
                videoCreatives = nvl(videoCreatives,
                        getVideoCreativesByBanners(data.getBannerIds(), data.getCreativesByBannerId()));
            }
            target.withVideoCreatives(videoCreatives);
        }
    }

    public List<String> getGenresAndCategories(GroupType groupType, FrontendData data, Map<Long, Goal> cryptaSegments,
                                               List<RetargetingCondition> retargetingConditions, List<Rule> rules) {
        groupType = getCorrectGroupType(groupType);
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(GENRES_AND_CATEGORIES)) {
            var genres = ifNotNull(data, d -> ifNotNull(d.getConditions(),
                    conditions -> getContentCategories(conditions, cryptaSegments,
                            GENRES_AND_CATEGORIES_GOAL_TYPES)));
            genres = nvl(genres, getContentCategories(retargetingConditions,
                    cryptaSegments, GENRES_AND_CATEGORIES_GOAL_TYPES));
            if (CollectionUtils.isEmpty(genres)) {
                genres = getContentCategories(rules,
                        cryptaSegments);
            }
            return genres;
        }
        return null;
    }

    private void addAudioCreatives(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(AUDIO_CREATIVES)) {
            List<AudioCreative> audioCreatives = ifNotNull(data.getFrontendData(), FrontendData::getAudioCreatives);
            audioCreatives = nvl(audioCreatives, getAudioCreatives(data.getBannerIds(), data.getCreativesByBannerId()));
            target.withAudioCreatives(audioCreatives);
        }
    }

    private void addCryptaGroups(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(CRYPTA_GROUPS)) {
            List<CryptaGroup> cryptaGroups = ifNotNull(data.getFrontendData(), d -> ifNotNull(d.getConditions(),
                    conditions -> getCryptaGroups(conditions, data.getGoalIdToCryptaGoalMapping())));
            cryptaGroups = nvl(cryptaGroups,
                    getCryptaGroups(data.getRetargetingConditions(), data.getGoalIdToCryptaGoalMapping()));
            target.withCryptaGroups(cryptaGroups);
        }
    }

    private void addAudienceGroups(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(AUDIENCE_GROUPS)) {
            List<AudienceGroup> audienceGroups = ifNotNull(data.getFrontendData(),
                    d -> ifNotNull(d.getConditions(), this::getAudienceGroups));
            audienceGroups = nvl(audienceGroups, getAudienceGroups(data.getRetargetingConditions()));
            target.withAudienceGroups(audienceGroups);
        }
    }

    private void addGenresAndCategories(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(GENRES_AND_CATEGORIES)) {
            var genres = ifNotNull(data.getFrontendData(), d -> ifNotNull(d.getConditions(),
                    conditions -> getContentCategories(conditions, data.getGoalIdToCryptaGoalMapping(),
                            GENRES_AND_CATEGORIES_GOAL_TYPES)));
            genres = nvl(genres, getContentCategories(data.getRetargetingConditions(),
                    data.getGoalIdToCryptaGoalMapping(), GENRES_AND_CATEGORIES_GOAL_TYPES));
            if (CollectionUtils.isEmpty(genres)) {
                genres = getContentCategories(data.getAdGroup().getContentCategoriesRetargetingConditionRules(),
                        data.getGoalIdToCryptaGoalMapping());
            }
            target.withGenresAndCategories(genres);
        }
    }

    private void addExcludedBrandSafetyCategories(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(EXCLUDED_BRAND_SAFETY_CATEGORIES)) {
            List<String> brandSafetyCategories = null;
            if (data.getFrontendData() == null || data.getFrontendData().getExcludedBsCategories() == null) {
                brandSafetyCategories = getContentCategories(
                        data.getRetargetingConditions(), data.getGoalIdToCryptaGoalMapping(), BRAND_SAFETY_GOAL_TYPES);
            } else {
                brandSafetyCategories = data.getFrontendData().getExcludedBsCategories();
            }
            var brandSafetyCategoriesToAdd = brandSafetyCategories.stream()
                    .map(c -> c.replace(BRAND_SAFETY_KEYWORD_NAME, BRAND_SAFETY_KEYWORD_TYPE))
                    .collect(toList());
            target.withExcludedBsCategories(brandSafetyCategoriesToAdd);
        }
    }

    private void addPageBlocks(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(PAGE_BLOCKS)) {
            List<PageBlock> pageBlocks = ifNotNull(data.getFrontendData(), FrontendData::getPageBlocks);
            pageBlocks = nvl(pageBlocks, convertPageBlocks(data.getPageBlocks()));
            target.withPageBlocks(pageBlocks);
        }
    }

    private void addExcludedPageBlocks(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(EXCLUDED_PAGE_BLOCKS)) {
            List<PageBlock> pageBlocks = convertPageBlocks(data.getExcludedPageBlocks());
            target.withExcludedPageBlocks(pageBlocks);
        }
    }

    private void addMainPageTrafficType(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(MAIN_PAGE_TRAFFIC_TYPE)) {
            target.withMainPageTrafficType(data.getMainPageTrafficType());
        }
    }

    private void addPlatformCorrections(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(PLATFORM_CORRECTIONS)) {
            PlatformCorrections platformCorrections =
                    ifNotNull(data.getFrontendData(), FrontendData::getPlatformCorrections);
            platformCorrections = nullableNvl(platformCorrections,
                    getPlatformCorrections(data.getBidModifierDesktop(), data.getBidModifierMobile()));
            target.withPlatformCorrections(platformCorrections);
        }
    }

    private void addProfileCorrections(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(PROFILE_CORRECTIONS)) {
            List<ProfileCorrection> profileCorrections =
                    ifNotNull(data.getFrontendData(), FrontendData::getProfileCorrections);
            profileCorrections = nullableNvl(profileCorrections,
                    ifNotNull(data.getBidModifierDemographics(), this::convertToProfileCorrections));
            target.withProfileCorrections(profileCorrections);
        }
    }

    private void addCorrections(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(CORRECTIONS)) {
            target.withCorrections(data.getCorrections());
        }
    }

    private void addProjectParameters(Target target, GroupType groupType, AdGroupData data) {
        if (FIELDS_BY_GROUP_TYPES.get(groupType).contains(PROJECT_PARAMETERS)) {
            target.withProjectParameters(data.getProjectParameters());
        }
    }

    private boolean haveMetrikaGoals(List<RetargetingCondition> rtConditions) {
        return StreamEx.of(rtConditions)
                .flatMap(rc -> StreamEx.of(rc.getRules()))
                .flatMap(rule -> StreamEx.of(rule.getGoals()))
                .anyMatch(goal -> goal.getType().isMetrika() && goal.getType() != AUDIENCE);
    }

    private Set<Integer> getRegions(@Nullable Set<Integer> campaignGeo, @Nullable List<Long> adGroupGeo) {
        Set<Integer> regions;
        if (isNotEmpty(adGroupGeo)) {
            regions = listToSet(adGroupGeo, Long::intValue);
        } else if (isNotEmpty(campaignGeo)) {
            regions = campaignGeo;
        } else {
            regions = emptySet();
        }
        return regions;
    }

    @Nullable
    private PlatformCorrections getPlatformCorrections(@Nullable BidModifierDesktop bidModifierDesktop,
                                                       @Nullable BidModifierMobile bidModifierMobile) {
        if (bidModifierDesktop != null || bidModifierMobile != null) {
            return PlatformCorrections.builder()
                    .withDesktop(ifNotNull(bidModifierDesktop, b -> b.getDesktopAdjustment().getPercent()))
                    .withMobile(ifNotNull(bidModifierMobile, b -> b.getMobileAdjustment().getPercent()))
                    .withMobileOsType(ifNotNull(
                            bidModifierMobile,
                            b -> ifNotNull(
                                    b.getMobileAdjustment().getOsType(),
                                    osType -> MobileOsType.valueOf(osType.name())
                            ))
                    )
                    .build();
        }
        return null;
    }

    public static Set<BlockSize> getAllowedBlockSizes(GroupType groupType, MainPageTrafficType mainPageTrafficType) {
        if (groupType == BANNER_IN_GEO_APPS) {
            return ALLOWED_BLOCK_SIZES_FOR_CPM_GEOPRODUCT;
        }
        if (groupType != MAIN_PAGE_AND_NTP || mainPageTrafficType == null) {
            return ALLOWED_BLOCK_SIZES;
        }
        switch (mainPageTrafficType) {
            case ALL:
                return ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_ALL_TYPES;
            case MOBILE:
                return ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_MOBILE;
            case DESKTOP:
                return ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_DESKTOP;
            case BROWSER_NEW_TAB:
                return ALLOWED_BLOCK_SIZES_FOR_FRONTPAGE_BROWSER_NEW_TAB;
        }
        throw new IllegalArgumentException();
    }

    private BlockSize getBlockSizeByCreative(Creative creative) {
        if (creative == null || creative.getWidth() == null || creative.getHeight() == null) {
            return null;
        }

        return new BlockSize(creative.getWidth().intValue(), creative.getHeight().intValue());
    }

    private BlockSize getBlockSizeByCreativeSize(CreativeSize creativeSize) {
        if (creativeSize == null || creativeSize.getWidth() == null || creativeSize.getHeight() == null) {
            return null;
        }

        return new BlockSize(creativeSize.getWidth().intValue(), creativeSize.getHeight().intValue());
    }

    private static boolean isCpmType(CampaignType type) {
        // type == null, когда кампании еще нет, сейчас это используется только для тачевого интерфейса, размеры
        // блоков в инветори в этом случае передавать нужно
        return type == null || type == CampaignType.CPM_DEALS || type == CampaignType.CPM_BANNER
                || type == CampaignType.CPM_YNDX_FRONTPAGE
                || type == CampaignType.CPM_PRICE;
    }

    public static List<CryptaGroup> getCryptaGroups(List<RetargetingCondition> rtConditions,
                                              Map<Long, Goal> goalIdToCryptaGoalMapping) {
        ImmutableSet<GoalType> types =
                ImmutableSet.of(GoalType.INTERESTS, GoalType.FAMILY, GoalType.SOCIAL_DEMO, GoalType.BEHAVIORS);
        return getCryptaGroups(rtConditions, goalIdToCryptaGoalMapping, types);
    }

    private static List<CryptaGroup> getCryptaGroups(List<RetargetingCondition> rtConditions,
                                              Map<Long, Goal> goalIdToCryptaGoalMapping, Set<GoalType> types) {
        StreamEx<Rule> rules = StreamEx.of(rtConditions)
                .flatMap(rc -> StreamEx.of(rc.getRules()))
                .filter(Objects::nonNull);
        return getCryptaGroups(rules, goalIdToCryptaGoalMapping, types);
    }

    private static List<CryptaGroup> getCryptaGroups(StreamEx<Rule> rules,
                                              Map<Long, Goal> goalIdToCryptaGoalMapping, Set<GoalType> types) {
        StreamEx<CryptaGroup> cryptaGroups = rules
                .map(r -> new CryptaGroup(
                        StreamEx.of(r.getGoals())
                                .filter(g -> types.contains(g.getType()))
                                .map(g -> goalIdToCryptaGoalMapping.get(g.getId()))
                                .flatMap(g -> (getCryptaSegments(g, r.getInterestType()).stream()))
                                .toSet()))
                .filter(cryptaGroup -> !cryptaGroup.getSegments().isEmpty());
        return cryptaGroups.toList();
    }

    private static List<String> getCryptaSegments(Goal g, CryptaInterestType interestType) {
        List<String> list = new ArrayList<>();

        if ((interestType == long_term || interestType == all || interestType == null)) {
            addCryptaSegment(list, g.getKeyword(), g.getKeywordValue());
        }

        if ((interestType == short_term || interestType == all || interestType == null)) {
            addCryptaSegment(list, g.getKeywordShort(), g.getKeywordValueShort());
        }

        return list;
    }

    private static void addCryptaSegment(List<String> list, String keyword, String keywordValue) {
        if (!StringUtils.isEmpty(keyword) && !UNDEFINED_KEYWORD.equals(keyword)) {
            String segment = keyword + ":" + keywordValue;
            list.add(segment);

            // В первом этапе охватного продукта в интерфейсе будет всего 4е дохода.
            // Но в прогнозаторе и в транспорте к БК нужно отдавать именно Средний доход как два значения.
            // https://wiki.yandex-team.ru/users/aliho/projects/direct/crypta/#diapazondljaobshhixpol/vozrast/doxod
            if (segment.equals(INCOME_B1)) {
                list.add(INCOME_B2);
            }
        }
    }

    private List<AudienceGroup> getAudienceGroups(List<RetargetingCondition> rtConditions) {
        StreamEx<Rule> rules = StreamEx.of(rtConditions)
                .flatMap(t -> StreamEx.of(t.getRules()))
                .filter(Objects::nonNull);
        StreamEx<AudienceGroup> audienceGroups = rules.map(this::makeAudienceGroupFromRule)
                .filter(t -> !t.getSegments().isEmpty());
        return audienceGroups.toList();
    }

    private AudienceGroup makeAudienceGroupFromRule(Rule someRule) {
//todo пока решили отправлять всю метрику в инвентори, окончательно разобраться в рамках DIRECT-104384
        Function<Goal, String> makeAudienceSegmentFromGoal =
                t -> t.getType().isMetrika() || t.getType() == GoalType.LAL_SEGMENT ? t.getId().toString() : null;
//                t -> t.getType() == AUDIENCE ? t.getId().toString() : null;
        return new AudienceGroup(convertRuleTypeToAudienceGroupingType(someRule.getType()),
                StreamEx.of(someRule.getGoals())
                        .map(makeAudienceSegmentFromGoal)
                        .filter(Objects::nonNull)
                        .toSet());
    }

    private List<String> getContentCategories(List<Rule> contentCategoriesRetargetingConditionRules,
                                              Map<Long, Goal> goalIdToCryptaGoalMapping) {
        if (CollectionUtils.isEmpty(contentCategoriesRetargetingConditionRules)) {
            return null;
        }
        var cryptaGroups = getCryptaGroups(StreamEx.of(contentCategoriesRetargetingConditionRules),
                goalIdToCryptaGoalMapping, GENRES_AND_CATEGORIES_GOAL_TYPES);
        return flatMap(mapList(cryptaGroups, CryptaGroup::getSegments), identity());
    }

    private List<String> getContentCategories(List<RetargetingCondition> retargetingConditions,
                                              Map<Long, Goal> goalIdToCryptaGoalMapping, Set<GoalType> goalTypes) {
        var cryptaGroups = getCryptaGroups(retargetingConditions, goalIdToCryptaGoalMapping, goalTypes);
        return flatMap(mapList(cryptaGroups, CryptaGroup::getSegments), identity());
    }

    private List<VideoCreative> getVideoCreativesByBanners(List<Long> bannersIds, Map<Long,
            Creative> creativesByBannerId) {
        return StreamEx.of(bannersIds)
                .map(creativesByBannerId::get)
                .filter(Objects::nonNull)
                // Оверлеи не участвуют в прогнозе
                .filter(c -> c.getType() != CreativeType.CPM_OVERLAY)
                .map(this::getVideoCreative)
                .toList();
    }

    private VideoCreative getVideoCreative(Creative creative) {
        int durationMillis = nvl(creative.getDuration(), 0L).intValue() * 1000;
        var blockSize = (creative.getWidth() != null && creative.getHeight() != null)
        ? new BlockSize(creative.getWidth().intValue(), creative.getHeight().intValue()) : null;
        return new VideoCreative(creative.getId(), durationMillis, blockSize, singleton(VIDEO_PROPORTION_16_9));
    }

    private List<VideoCreative> getOutdoorVideoCreativesByBanners(List<Long> bannersIds, Map<Long,
            Creative> creativesByBannerId) {
        return StreamEx.of(bannersIds)
                .map(creativesByBannerId::get)
                .filter(Objects::nonNull)
                .map(this::getVideoCreativeWithAdditionalData)
                .toList();
    }

    private VideoCreative getVideoCreativeWithAdditionalData(Creative creative) {
        return convertCreativeWithAdditionalData(
                creative,
                this::convertToSize,
                null
        );
    }

    @Nullable
    private BlockSize convertToSize(VideoFormat format) {
        if (format.getWidth() == null || format.getHeight() == null) {
            return null;
        }

        return new BlockSize(format.getWidth(), format.getHeight());
    }

    private List<VideoCreative> getIndoorVideoCreativesByBanners(List<Long> bannersIds, Map<Long,
            Creative> creativesByBannerId) {
        return StreamEx.of(bannersIds)
                .map(creativesByBannerId::get)
                .filter(Objects::nonNull)
                .map(this::convertVideoCreative)
                .toList();
    }

    private VideoCreative convertVideoCreative(Creative creative) {
        return convertCreativeWithAdditionalData(
                creative,
                null,
                AdGroupDataConverter::convertToProportion
        );
    }

    @Nullable
    private static BlockSize convertToProportion(VideoFormat format) {
        Integer width = format.getWidth();
        Integer height = format.getHeight();

        if (width == null || height == null) {
            return null;
        }

        int gcd = gcd(width, height);
        if (gcd == 0) {
            gcd = 1;
        }

        int proportionWidth = width / gcd;
        int proportionHeight = height / gcd;

        return new BlockSize(proportionWidth, proportionHeight);
    }


    private List<AudioCreative> getAudioCreatives(List<Long> bannersIds, Map<Long, Creative> creativesByBannerId) {
        return StreamEx.of(bannersIds)
                .map(creativesByBannerId::get)
                .filter(Objects::nonNull)
                .map(this::convertToAudioCreative)
                .toList();
    }

    private AudioCreative convertToAudioCreative(Creative creative) {
        int durationMillis = creative.getDuration().intValue() * 1000;
        return new AudioCreative(durationMillis,
                new BlockSize(creative.getWidth().intValue(), creative.getHeight().intValue()));
    }

    private List<ProfileCorrection> convertToProfileCorrections(BidModifierDemographics bidModifierDemographics) {
        return StreamEx.of(bidModifierDemographics.getDemographicsAdjustments())
                .map(demography -> ProfileCorrection.builder()
                        .withGender(ifNotNull(demography.getGender(),
                                CommonConverters::convertToProfileCorrectionGender))
                        .withAge(ifNotNull(demography.getAge(), CommonConverters::convertToProfileCorrectionAge))
                        .withCorrection(demography.getPercent())
                        .build())
                .toList();
    }

    protected enum Field {
        REGIONS,
        DOMAINS,
        EXCLUDED_DOMAINS,
        SSP,
        EXCLUDED_SSP,
        BLOCK_SIZES,
        VIDEO_CREATIVES,
        AUDIO_CREATIVES,
        CRYPTA_GROUPS,
        AUDIENCE_GROUPS,
        GENRES_AND_CATEGORIES,
        EXCLUDED_BRAND_SAFETY_CATEGORIES,
        PAGE_BLOCKS,
        EXCLUDED_PAGE_BLOCKS,
        MAIN_PAGE_TRAFFIC_TYPE,
        PLATFORM_CORRECTIONS,
        PROFILE_CORRECTIONS,
        CORRECTIONS,
        TARGET_TAGS,
        ORDER_TAGS,
        ENABLE_NON_SKIPPABLE_VIDEO,
        IS_DEFAULT_GROUP,
        PROJECT_PARAMETERS,
    }
}
