package ru.yandex.direct.core.entity.inventori.service.type;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.inventori.model.request.AudienceGroup;
import ru.yandex.direct.inventori.model.request.BlockSize;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.inventori.model.request.ProfileCorrection;
import ru.yandex.direct.inventori.model.request.VideoCreative;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.inventori.model.request.GroupType.AUDIO;
import static ru.yandex.direct.inventori.model.request.GroupType.BANNER;
import static ru.yandex.direct.inventori.model.request.GroupType.BANNER_IN_GEO_APPS;
import static ru.yandex.direct.inventori.model.request.GroupType.GEO_PIN;
import static ru.yandex.direct.inventori.model.request.GroupType.INDOOR;
import static ru.yandex.direct.inventori.model.request.GroupType.MAIN_PAGE_AND_NTP;
import static ru.yandex.direct.inventori.model.request.GroupType.OUTDOOR;
import static ru.yandex.direct.inventori.model.request.GroupType.UNKNOWN;
import static ru.yandex.direct.inventori.model.request.GroupType.VIDEO;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public final class CommonConverters {
    private static final Logger log = LoggerFactory.getLogger(CommonConverters.class);

    private CommonConverters() {
    }

    public static List<ru.yandex.direct.inventori.model.request.PageBlock> convertPageBlocks(List<PageBlock> pageTargets) {
        return EntryStream.of(StreamEx.of(pageTargets)
                .groupingBy(PageBlock::getPageId))
                .mapKeyValue((pageId, pageBlocks) -> {
                    var values = mapList(pageBlocks, PageBlock::getImpId);
                    if (values.size() == 1 && values.get(0) == null) {
                        values = null;
                    }
                    return new ru.yandex.direct.inventori.model.request.PageBlock(pageId, values);
                })
                .toList();
    }

    public static VideoCreative convertCreativeWithAdditionalData(Creative creative,
                                                                  @Nullable Function<VideoFormat, BlockSize> sizeConverter,
                                                                  @Nullable Function<VideoFormat, BlockSize> proportionsConverter) {
        AdditionalData additionalData = creative.getAdditionalData();
        checkState(additionalData != null, "Video creative additional data must not be null");
        BigDecimal duration = additionalData.getDuration();
        checkState(duration != null, "Video creative duration must not be null");
        List<VideoFormat> formats = additionalData.getFormats();
        checkState(formats != null, "Video creative formats must not be null");

        int inventoriDuration = duration.multiply(BigDecimal.valueOf(1000)).intValue();

        //  Переделала множественное на единичное потому что продукт outdoor, где это использовалось закрыли,
        //  API inventori изменили, а outdoor в ближайшее время планируется оторвать из Директа
        //  https://st.yandex-team.ru/DIRECT-155280
        BlockSize inventoriSize = null;
        if (sizeConverter != null && !formats.isEmpty()) {
            VideoFormat format = formats.get(0);
            if (format.getWidth() != null && format.getHeight() != null) {
                inventoriSize = new BlockSize(format.getWidth().intValue(), format.getHeight().intValue());
            }
        }

        Set<BlockSize> inventoriProportions = null;
        if (proportionsConverter != null) {
            inventoriProportions = StreamEx.of(formats)
                    .map(proportionsConverter)
                    .filter(Objects::nonNull)
                    .toSet();
        }

        return new VideoCreative(inventoriDuration, inventoriSize, inventoriProportions);
    }

    public static GroupType convertGroupType(AdGroupType adGroupType) {
        switch (adGroupType) {
            case BASE: // Для ТГО в тачевом интерфейсе сейчас в инвентори передаем тип BANNER
            case CPM_BANNER:
                return BANNER;
            case CPM_AUDIO:
                return AUDIO;
            case CPM_VIDEO:
                return VIDEO;
            case CPM_OUTDOOR:
                return OUTDOOR;
            case CPM_INDOOR:
                return INDOOR;
            case CPM_YNDX_FRONTPAGE:
                return MAIN_PAGE_AND_NTP;
            case CPM_GEOPRODUCT:
                return BANNER_IN_GEO_APPS;
            case CPM_GEO_PIN:
                return GEO_PIN;
        }
        log.warn(
                "AdGroupDataConverter.getGroupType: failed to suggest group type for adGroupType = {}",
                adGroupType);

        return UNKNOWN;
    }

    public static AudienceGroup.GroupingType convertRuleTypeToAudienceGroupingType(RuleType someRuleType) {
        switch (someRuleType) {
            case OR:
                return AudienceGroup.GroupingType.ANY;
            case ALL:
                return AudienceGroup.GroupingType.ALL;
            case NOT:
                return AudienceGroup.GroupingType.NOT_ANY;
        }
        return AudienceGroup.GroupingType.ANY;
    }

    public static ProfileCorrection.Gender convertToProfileCorrectionGender(GenderType bidModifierDemographicsGender) {
        switch (bidModifierDemographicsGender) {
            case MALE:
                return ProfileCorrection.Gender.MALE;
            case FEMALE:
                return ProfileCorrection.Gender.FEMALE;
            default:
                throw new IllegalStateException("Unsupported gender value: " + bidModifierDemographicsGender);
        }
    }

    public static ProfileCorrection.Age convertToProfileCorrectionAge(AgeType bidModifierDemographicsAge) {
        switch (bidModifierDemographicsAge) {
            case _0_17:
                return ProfileCorrection.Age._0_17;
            case _18_24:
                return ProfileCorrection.Age._18_24;
            case _25_34:
                return ProfileCorrection.Age._25_34;
            case _35_44:
                return ProfileCorrection.Age._35_44;
            case _45_:
            case _45_54:
                return ProfileCorrection.Age._45_54;
            case _55_:
                return ProfileCorrection.Age._55_;
            default:
                throw new IllegalStateException("Unsupported age value: " + bidModifierDemographicsAge);
        }
    }
}
