package ru.yandex.direct.core.entity.keyword.model;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

/**
 * Представляет ключевую фразу вместе с минус-фразами.
 */
@ParametersAreNonnullByDefault
public class KeywordWithMinuses {
    private static final Pattern MINUS_KEYWORD_DELIMITER = Pattern.compile("\\s+-");

    private final String plusKeyword;
    private final Set<String> minusKeywords; // сам список изменяем(!)

    private KeywordWithMinuses(String plusKeyword, Collection<String> minusKeywords) {
        this.plusKeyword = plusKeyword;
        this.minusKeywords = new LinkedHashSet<>(minusKeywords);
    }

    /**
     * Преобразует строку из формата {@code BIDS.PHRASE} в ключевую фразу и набор минус-фраз {@link KeywordWithMinuses}
     */
    public static KeywordWithMinuses fromPhrase(String keywordWithMinusesText) {
        if (keywordWithMinusesText.startsWith("-")) {
            // если плюс-фраза отсутствует, дополним строку пробелом, чтобы отработала логика split(" -")
            keywordWithMinusesText = " " + keywordWithMinusesText;
        }

        String[] split = MINUS_KEYWORD_DELIMITER.split(keywordWithMinusesText);
        List<String> minusKeywords = StreamEx.of(split).skip(1).toList();
        return new KeywordWithMinuses(split[0], minusKeywords);
    }

    /**
     * Добавляет переданную фразу к набору минус-фраз
     */
    public void addMinusKeyword(String minusKeyword) {
        this.minusKeywords.add(minusKeyword);
    }

    /**
     * Добавляет переданный список фраз к набору минус-фраз
     */
    public void addMinusKeywords(Collection<String> minusKeywords) {
        this.minusKeywords.addAll(minusKeywords);
    }

    public String getPlusKeyword() {
        return plusKeyword;
    }

    public List<String> getMinusKeywords() {
        return List.copyOf(minusKeywords);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(plusKeyword);
        minusKeywords.forEach(mk -> sb.append(" -").append(mk));
        return sb.toString();
    }
}
